#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
用法
----
1. 把 LLM 的聚合结果保存为 agg.json
2. python analyze_jsonl.py
"""
import json
import csv
from collections import Counter
import matplotlib.pyplot as plt
from pathlib import Path

# Derive stem from hardcoded input file
input_path = Path('agg_example.json')
input_stem = input_path.stem
print(f"Extracted filename stem (without extension): {input_stem}")

# 1. 读入聚合结果
with open(input_path, 'r', encoding='utf-8') as f:
    data = json.load(f)

intent_hist = data['intent_histogram']
action_hist = data['action_histogram']
merge_log   = data['merge_log']

# 2. 简易彩色终端柱状图
def bar(title: str, counter: Counter):
    """
    在终端里打印一个彩色水平柱状图，并保存为 PNG
    """
    GREEN = '\033[92m'
    RESET = '\033[0m'
    max_label = max(len(k) for k in counter) if counter else 0
    max_val   = max(counter.values()) if counter else 1
    print(f"\n{GREEN}{title}{RESET}")
    for k, v in counter.most_common():
        bar_len = int(v / max_val * 60)
        print(f"{k.ljust(max_label)} │{'█' * bar_len} {v}")

    # matplotlib 可视化
    labels, values = zip(*counter.most_common())
    fig, ax = plt.subplots()
    ax.barh(labels, values)
    ax.set_xlabel('Count')
    ax.set_title(title)
    plt.tight_layout()
    filename = f"{input_stem}_{title.lower().replace(' ', '_')}_bar.png"
    plt.savefig(filename)
    plt.close()
    print(f"Saved bar chart: {filename}")

bar("Intent Distribution", Counter(intent_hist))
bar("Action Distribution", Counter(action_hist))

# 3. ASCII 饼图
def pie(title: str, counter: Counter):
    """
    极简 ASCII 饼图（仅作概览），并保存为 PNG
    """
    total = sum(counter.values())
    if not total:
        return
    print(f"\n{title}")
    for k, v in counter.items():
        pct = v / total * 100
        print(f"{k:20} {pct:5.1f}% {'■' * int(pct/5)}")

    # matplotlib 可视化
    labels = list(counter.keys())
    sizes = list(counter.values())
    fig, ax = plt.subplots()
    ax.pie(sizes, labels=labels, autopct='%1.1f%%')
    ax.set_title(title)
    plt.tight_layout()
    filename = f"{input_stem}_{title.lower().replace(' ', '_')}_pie.png"
    plt.savefig(filename)
    plt.close()
    print(f"Saved pie chart: {filename}")

pie("Intent Distribution", Counter(intent_hist))
pie("Action Distribution", Counter(action_hist))

# 4. 把合并明细导出成 CSV（审计用）
with open(f"{input_stem}_merge_detail.csv", 'w', newline='', encoding='utf-8') as f:
    writer = csv.writer(f)
    writer.writerow(['Type', 'MergedLabel', 'OriginalIndex'])
    for t in ('intent', 'action'):
        for label, idx_list in merge_log[t].items():
            for idx in idx_list:
                writer.writerow([t, label, idx])

print(f"\n✅ 已生成 {input_stem}_merge_detail.csv，可导入 Excel / Pandas 进行进一步审计。")