import pytest

import torch
import torch.nn as nn

from nesim.losses.laplacian_pyramid.loss import LaplacianPyramidLoss
from nesim.utils.grid_size import find_rectangle_dimensions

linear_layer_possible_params = {
    "in_size": [64, 128],
    "out_size": [15, 512, 1024],
    "shrink_factor": [
        [2.0],
        [
            2.0,
            1.0,
            3.0,
        ],
    ],
}


@pytest.mark.parametrize("in_size", linear_layer_possible_params["in_size"])
@pytest.mark.parametrize("out_size", linear_layer_possible_params["out_size"])
@pytest.mark.parametrize("shrink_factor", linear_layer_possible_params["shrink_factor"])
def test_linear(in_size, out_size, shrink_factor):
    """
    Test the following:
    1. get_loss() should return a tensor
    2. get_loss() should return a 0-D tensor
    3. grid should give us a tensor of shape: [size.height, size.width, *]
    """
    linear_layer = nn.Linear(in_size, out_size)
    size = find_rectangle_dimensions(area=out_size)

    loss_calculator = LaplacianPyramidLoss(
        layer=linear_layer, factor_h=shrink_factor, factor_w=shrink_factor, device="cpu"
    )
    loss = loss_calculator.get_loss()

    # 1
    assert torch.is_tensor(loss), "Expected loss to be a torch tensor"
    # 2
    assert len(loss.shape) == 0
    # 3
    assert loss_calculator.grid_container.grid.shape[:2] == (size.height, size.width)
