import torch
from torchtyping import TensorType


def pearsonr(
    x: TensorType["batch_size", "n"], y: TensorType["batch_size", "n"]
) -> torch.Tensor:
    """
    Mimics `scipy.stats.pearsonr` with support for batched inputs
    Arguments
    ---------
    x : torch.Tensor
        Batched input of shape (batch_size, n)
    y : torch.Tensor
        Batched input of shape (batch_size, n)
    Returns
    -------
    r_val : torch.Tensor
        Pearson correlation coefficients between x and y, of shape (batch_size,)

    Scipy docs ref:
        XXXX

    Scipy code ref:
        XXXX
    Example:
        >>> x = np.random.randn(100)
        >>> y = np.random.randn(100)
        >>> sp_corr = scipy.stats.pearsonr(x, y)[0]
        >>> th_corr = pearsonr(torch.from_numpy(x), torch.from_numpy(y))
        >>> np.allclose(sp_corr, th_corr)
    """
    assert x.ndim == y.ndim == 2, "Both x and y should be 2D tensors."
    assert x.size(0) == y.size(0), "The batch sizes of x and y should be the same."
    assert x.size(1) == y.size(1), "The 'n' dimensions of x and y should be the same."

    mean_x = torch.mean(x, dim=1, keepdim=True)
    mean_y = torch.mean(y, dim=1, keepdim=True)
    xm = x - mean_x
    ym = y - mean_y
    r_num = torch.sum(xm * ym, dim=1)
    r_den = torch.norm(xm, dim=1) * torch.norm(ym, dim=1)
    r_val = r_num / r_den
    return r_val
