import torch


def normalize_tensor(tensor, epsilon=1e-8):
    """
    normalize your PyTorch tensor between 0 and 1
    """
    min_val = torch.min(tensor)
    max_val = torch.max(tensor)
    normalized_tensor = (tensor - min_val) / (max_val - min_val + epsilon)
    return normalized_tensor


def z_score_normalize(tensor, dim=0):
    """
    Z-score normalizes a PyTorch tensor along the specified dimension.

    Args:
        tensor (torch.Tensor): Input tensor to be normalized.
        dim (int): Dimension along which normalization is performed. Default is 0.

    Returns:
        torch.Tensor: Z-score normalized tensor.
    """
    # Calculate mean and standard deviation along the specified dimension
    mean = torch.mean(tensor, dim=dim, keepdim=True)
    std = torch.std(tensor, dim=dim, keepdim=True)

    # Z-score normalization
    normalized_tensor = (tensor - mean) / std

    return normalized_tensor