from PIL import Image
from tqdm import tqdm


def make_grid_from_list_of_images(images, height=32, width=32):
    """
    Creates a grid of images from a list of PIL images.

    Args:
        images (list): A list of PIL images.
        height (int): The height (number of images) of the grid. Default is 32.
        width (int): The width (number of images) of the grid. Default is 32.

    Returns:
        grid_image (PIL.Image.Image): The grid image containing the input images.

    Raises:
        AssertionError: If the number of images is not equal to height * width.

    """

    assert (
        len(images) == height * width
    ), "Number of images is not equal to height * width."

    image_width, image_height = images[0].size
    grid_width = image_width * width
    grid_height = image_height * height

    grid_image = Image.new("RGB", (grid_width, grid_height))

    for i, image in tqdm(enumerate(images)):
        row = i // width
        col = i % width
        grid_image.paste(image, (col * image_width, row * image_height))

    return grid_image
