import pytest

import torch
import torch.nn as nn

from nesim.losses.ring.loss import RingLoss1D

device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

linear_layer_possible_params = {
    "in_size": [64, 128],
    "out_size": [15, 512, 1024],
}


@pytest.mark.parametrize("in_size", linear_layer_possible_params["in_size"])
@pytest.mark.parametrize("out_size", linear_layer_possible_params["out_size"])
def test_linear(in_size, out_size):
    """
    Test the following:
    1. get_loss() should return a tensor
    2. get_loss() should return a 0-D tensor
    """
    linear_layer = nn.Linear(in_size, out_size)

    loss_calculator = RingLoss1D(
        layer=linear_layer, freq_inner=5, freq_outer=6
    )
    loss = loss_calculator.get_loss()

    # 1
    assert torch.is_tensor(loss), "Expected loss to be a torch tensor"
    # 2
    assert len(loss.shape) == 0


@pytest.mark.parametrize("in_size", linear_layer_possible_params["in_size"])
@pytest.mark.parametrize("out_size", linear_layer_possible_params["out_size"])
@pytest.mark.parametrize("n_training_steps", [2, 10, 20])
def test_training_loop(in_size, out_size, n_training_steps):
    """
    Test the following:
    1. get_loss_fast() should return the same result as get_loss_original()
    2. on a small training loop, both losses should go down with very close values
    """
    linear_layer = nn.Linear(in_size, out_size)

    loss_calculator = RingLoss1D(
        layer=linear_layer, freq_inner=5, freq_outer=6
    )
    optimizer = torch.optim.Adam(linear_layer.parameters(), lr=1e-3)

    losses = []
    for train_step_idx in range(n_training_steps):
        optimizer.zero_grad()
        loss = loss_calculator.get_loss()
        loss.backward()
        optimizer.step()
        losses.append(loss.item())

    if len(losses) > 0:
        assert losses[-1] < losses[0]
