import math
import sys

import torch

from clstool import build_evaluator   
from clstool.utils.misc import update, reduce_dict, MetricLogger, SmoothedValue


def train_one_epoch(model, criterion, data_loader, optimizer, lr_scheduler, device, epoch: int, max_norm: float = 0,
                    scaler=None, print_freq: int = 10, need_targets: bool = False):
    model.train()
    criterion.train()
    n_steps = len(data_loader)

    metric_logger = MetricLogger(delimiter='  ')
    metric_logger.add_meter('lr', SmoothedValue(window_size=1, fmt='{value:.6f}'))
    metric_logger.add_meter('class_error', SmoothedValue(window_size=1, fmt='{value:.2f}'))
    header = 'Epoch: [{}]'.format(epoch)

    for batch_idx, (samples, targets, protected, filenames) in enumerate(metric_logger.log_every(data_loader, print_freq, header)):
        samples = samples.to(device)
        targets = targets.to(device)

        with torch.cuda.amp.autocast(enabled=scaler is not None):
            if need_targets:
                outputs = model(samples, targets)
            else:
                outputs = model(samples)

            loss_dict = criterion(outputs, targets, training=True)
            weight_dict = criterion.weight_dict
            losses = sum(loss_dict[k] * weight_dict[k] for k in loss_dict.keys() if k in weight_dict)

        loss_dict_reduced = reduce_dict(loss_dict)
        loss_dict_reduced_unscaled = {f'{k}_unscaled': v for k, v in loss_dict_reduced.items()}
        loss_dict_reduced_scaled = {k: v * weight_dict[k] for k, v in loss_dict_reduced.items() if k in weight_dict}
        losses_reduced_scaled = sum(loss_dict_reduced_scaled.values())

        loss_value = losses_reduced_scaled.item()

        if not math.isfinite(loss_value):
            print('Loss is {}, stopping training'.format(loss_value))
            print(loss_dict_reduced)
            sys.exit(1)

        update(optimizer, losses, model, max_norm, scaler)

        if hasattr(lr_scheduler, 'step_update'):
            lr_scheduler.step_update(epoch * n_steps + batch_idx)

        metric_logger.update(loss=loss_value, **loss_dict_reduced_scaled, **loss_dict_reduced_unscaled)
        metric_logger.update(lr=optimizer.param_groups[0]['lr'])
        if 'class_error' in loss_dict_reduced.keys():
            metric_logger.update(class_error=loss_dict_reduced['class_error'])

    lr_scheduler.step(epoch)

    metric_logger.synchronize_between_processes()
    print('Averaged stats:', metric_logger)

    stats = {k: meter.global_avg for k, meter in metric_logger.meters.items() if meter.count > 0}

    return stats

@torch.no_grad()
def evaluate(model, data_loader, criterion, device, args, print_freq=10, need_targets=False, amp=False):
    model.eval()
    criterion.eval()

    metric_logger = MetricLogger(delimiter='  ')
    metric_logger.add_meter('class_error', SmoothedValue(window_size=1, fmt='{value:.2f}'))
    header = 'Test:'

    evaluator_0 = build_evaluator(args)
    evaluator_1 = build_evaluator(args)
    evaluator_2 = build_evaluator(args)
    evaluator_3 = build_evaluator(args)
    evaluator_4 = build_evaluator(args)

    for samples, targets, protected, filenames in metric_logger.log_every(data_loader, print_freq, header):
        samples = samples.to(device)
        targets = targets.to(device)
        protected = protected.to(device)

        with torch.cuda.amp.autocast(enabled=amp):
            if need_targets:
                outputs = model(samples, targets)
            else:
                outputs = model(samples)

            loss_dict = criterion(outputs, targets, training=False)

        weight_dict = criterion.weight_dict
        loss_dict_reduced = reduce_dict(loss_dict)
        loss_dict_reduced_unscaled = {f'{k}_unscaled': v for k, v in loss_dict_reduced.items()}
        loss_dict_reduced_scaled = {k: v * weight_dict[k] for k, v in loss_dict_reduced.items() if k in weight_dict}

        metric_logger.update(loss=sum(loss_dict_reduced_scaled.values()),
                             **loss_dict_reduced_scaled,
                             **loss_dict_reduced_unscaled,
                             )
        metric_logger.update(class_error=loss_dict_reduced['class_error'])

        evaluator_0.update(outputs, targets, protected[:,0])
        evaluator_1.update(outputs, targets, protected[:,1])
        evaluator_2.update(outputs, targets, protected[:,2])
        evaluator_3.update(outputs, targets, protected[:,3])
        evaluator_4.update(outputs, targets, protected[:,4])

    metric_logger.synchronize_between_processes()
    print('Averaged stats:', metric_logger)

    evaluator_0.synchronize_between_processes()
    evaluator_1.synchronize_between_processes()
    evaluator_2.synchronize_between_processes()
    evaluator_3.synchronize_between_processes()
    evaluator_4.synchronize_between_processes()

    evaluator_0.summarize()
    evaluator_1.summarize()
    evaluator_2.summarize()
    evaluator_3.summarize()
    evaluator_4.summarize()
    
    eval_0 = ', '.join(f"{key}: {value}" for key, value in evaluator_0.eval.items())
    eval_1 = ', '.join(f"{key}: {value}" for key, value in evaluator_1.eval.items())
    eval_2 = ', '.join(f"{key}: {value}" for key, value in evaluator_2.eval.items())
    eval_3 = ', '.join(f"{key}: {value}" for key, value in evaluator_3.eval.items())
    eval_4 = ', '.join(f"{key}: {value}" for key, value in evaluator_4.eval.items())
    
    ACC = (evaluator_0.eval['acc'] + evaluator_1.eval['acc'] + evaluator_2.eval['acc'] + evaluator_3.eval['acc'] + evaluator_4.eval['acc'])/5
    DP = (evaluator_0.eval['dp'] + evaluator_1.eval['dp'] + evaluator_2.eval['dp'] + evaluator_3.eval['dp'] + evaluator_4.eval['dp'])/5
    EOPP = (evaluator_0.eval['eopp'] + evaluator_1.eval['eopp'] + evaluator_2.eval['eopp'] + evaluator_3.eval['eopp'] + evaluator_4.eval['eopp'])/5
    EODD = (evaluator_0.eval['eodd'] + evaluator_1.eval['eodd'] + evaluator_2.eval['eodd'] + evaluator_3.eval['eodd'] + evaluator_4.eval['eodd'])/5
    AOD = (evaluator_0.eval['aod'] + evaluator_1.eval['aod'] + evaluator_2.eval['aod'] + evaluator_3.eval['aod'] + evaluator_4.eval['aod'])/5

    with open(f'out_eval.txt', 'a') as file:
        print(f'model:{args.model}', file=file)
        print(f'{eval_0}  sa_{args.sub_attrs[0]}', file=file)
        print(f'{eval_1}  sa_{args.sub_attrs[1]}', file=file)
        print(f'{eval_2}  sa_{args.sub_attrs[2]}', file=file)
        print(f'{eval_3}  sa_{args.sub_attrs[3]}', file=file)
        print(f'{eval_4}  sa_{args.sub_attrs[4]}', file=file)
        print('ACC: {:.2f}  DP: {:.2f}  EOPP: {:.2f}  EODD: {:.2f}  AOD: {:.2f}  Mean'.format(ACC, DP, EOPP, EODD, AOD), file=file)
        print(' ', file=file)
    
    stats = {k: meter.global_avg for k, meter in metric_logger.meters.items()}

    stats['eval'] = list(evaluator_0.eval.values())

    return stats, evaluator_0
