import collections
import os

import numpy as np

# The function to retrieve the rotation matrix changed from as_dcm to as_matrix in version 1.4
# We will use the version number for backcompatibility
import scipy
import yaml
from packaging import version
from PIL import Image

from habitat_sim.utils.common import quat_from_angle_axis
import math

import textwrap
from typing import Dict, List, Optional, Tuple, Union

import imageio
import tqdm
 
from enlighten.utils.path import *

from enlighten.utils.image_utils import try_cv2_import
cv2 = try_cv2_import()

from enlighten.utils.tensorboard_utils import TensorboardWriter

def BGR_mode(config):
    # BGR_mode = False
    # if config.get("single_scene"):
    #     if "replica" in config.get("scene_id"):
    #         BGR_mode = True
    # else:    
    #     if "replica" in config.get("dataset_path"):
    #         BGR_mode = True
    BGR_mode = True
    return BGR_mode
            
def generate_video(
    video_option: List[str],
    video_dir: Optional[str],
    images: List[np.ndarray],
    episode_id: Union[int, str],
    checkpoint_idx: int,
    metrics: Dict[str, float],
    tb_writer: TensorboardWriter,
    fps: int = 10,
) -> None:
    r"""Generate video according to specified information.

    Args:
        video_option: string list of "tensorboard" or "disk" or both.
        video_dir: path to target video directory.
        images: list of images to be converted to video.
        episode_id: episode id for video naming.
        checkpoint_idx: checkpoint index for video naming.
        metric_name: name of the performance metric, e.g. "spl".
        metric_value: value of metric.
        tb_writer: tensorboard writer object for uploading video.
        fps: fps for generated video.
    Returns:
        None
    """
    if len(images) < 1:
        return

    metric_strs = []
    for k, v in metrics.items():
        metric_strs.append(f"{k}={v:.2f}")

    video_name = f"episode={episode_id}-ckpt={checkpoint_idx}-" + "-".join(
        metric_strs
    )
    if "disk" in video_option:
        assert video_dir is not None
        images_to_video(images, video_dir, video_name)
    if "tensorboard" in video_option:
        tb_writer.add_video_from_np_images(
            f"episode{episode_id}", checkpoint_idx, images, fps=fps
        )

def remove_jpg(video_path):
    for file in os.listdir(video_path):
        if file.endswith(".jpg"):
            os.remove(os.path.join(video_path, file))

def images_to_video(
    images: List[np.ndarray],
    output_dir: str,
    video_name: str,
    fps: int = 10,
    quality: Optional[float] = 5,
    **kwargs,
):
    r"""Calls imageio to run FFMPEG on a list of images. For more info on
    parameters, see https://imageio.readthedocs.io/en/stable/format_ffmpeg.html
    Args:
        images: The list of images. Images should be HxWx3 in RGB order.
        output_dir: The folder to put the video in.
        video_name: The name for the video.
        fps: Frames per second for the video. Not all values work with FFMPEG,
            use at your own risk.
        quality: Default is 5. Uses variable bit rate. Highest quality is 10,
            lowest is 0.  Set to None to prevent variable bitrate flags to
            FFMPEG so you can manually specify them using output_params
            instead. Specifying a fixed bitrate using ‘bitrate’ disables
            this parameter.
    """
    assert 0 <= quality <= 10
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
    video_name = video_name.replace(" ", "_").replace("\n", "_") + ".mp4"
    writer = imageio.get_writer(
        os.path.join(output_dir, video_name),
        fps=fps,
        quality=quality,
        **kwargs,
    )
    print("Video created: {os.path.join(output_dir, video_name)}")
    for im in tqdm.tqdm(images):
        writer.append_data(im)
    writer.close()  

# my create video
def create_video(video_path, BGR_mode, video_name="video"):
    images = []
    for file in os.listdir(video_path):
        if file.endswith(".jpg"):
            #print(os.path.join(output_path, file))
            img = cv2.imread(os.path.join(video_path, file))
            # BGR to RGB
            if BGR_mode:
                img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
            images.append(img)

    if len(images) > 0:
        images_to_video(images, output_dir=video_path, video_name=video_name)
    else:
        print("Error: no images exist!")    

if __name__ == "__main__":
    create_video(video_path="~/enlighten/video", BGR_mode=True)