#include <iostream>
#include <fstream>
#include <vector>
#include <queue>
#include <unordered_set>
#include <limits>
#include <algorithm>
#include <map>
#include <set>
#include <chrono>
#include <omp.h>


// Function to perform BFS up to a given depth limit
int bfs_limited(const std::vector<std::unordered_set<int>>& adj, 
		               std::map<int, int> &neighbors,
			       int start, int depth_limit) {
    int n = adj.size();
    std::vector<bool> visited(n, false);
    std::queue<std::pair<int, int> > q;
    
    visited[start] = true;
    q.push(std::make_pair(start,0));
    
    while (!q.empty()) {
        auto it = q.front();
        q.pop();
        int d = it.second;
	int u = it.first;
	neighbors.insert(std::make_pair(u,d));
        if (d == depth_limit) continue;  // Stop going deeper than depth_limit
        
        for (int v : adj[u]) {
            if (visited[v] == false) {
                visited[v] = true;
                q.push(std::make_pair(v,d+1));
            }
        }
    }
    return 0;
}

int main(int argc, char* argv[]) {
    if (argc != 3) {
        std::cerr << "Usage: " << argv[0] << " <input_file> <output_file>\n";
        return 1;
    }
    
    std::string input_filename = argv[1];
    std::string output_filename = argv[2];
    
    std::ifstream infile(input_filename);
    if (!infile) {
        std::cerr << "Cannot open input file.\n";
        return 1;
    }
    

    // input all train edges
    int num_edges, num_nodes, num_queries, k;
    infile >> num_edges >> num_nodes >> num_queries >> k;
    std::cout << num_edges << ' ' << num_queries << std::endl;

    // Create adjacency list representation of the graph
    std::vector<std::unordered_set<int>> adj(num_nodes);
    std::cout << "done" << std::endl;
    int u, v;
    for (int i = 0; i < num_edges; ++i){
	std::cout << i << "\r" << std::flush;
        infile >> u >> v; 
        adj[u].insert(v);
        adj[v].insert(u);

    }

    int radius = (k + 1) / 2;
    std::cout << "radius = " << radius << std::endl;
    std::vector<int> scores(num_queries, -1);
    
    std::vector<std::pair<int, int> > queries(num_queries);
    // For each edge, compute capped shortest path
    auto start_time = std::chrono::high_resolution_clock::now();
    for (int i = 0; i < num_queries; ++i) {
	if (i % 10000 == 0) std::cout << i << "\r" << std::flush;
        int s;
        int t;
        infile >> s >> t;
        queries[i] = std::make_pair(s,t);
    }
    // maintain r-hop neighbors for each node
    /*
    std::cout << "computing 2-hop neighbors" << std::endl;
    std::vector<std::map<int, int>> r_hops(num_queries);
    int keep_limit = 10000;
    int cnt = 0;
    for(int i = 0; i < num_nodes; i++){
        if (i % 10000 == 0) std::cout << i << ' ' << cnt << "\r" << std::flush;
	std::map<int, int> tmp;
	tmp.clear();
	bfs_limited(adj, tmp, i, radius);
	if (tmp.size() >= keep_limit){
		r_hops[i] = tmp;
		cnt++;
	}
	else{
		r_hops[i].clear();

	}
    }
    */
//    adj.clear();
//    adj.shrink_to_fit();

    // use openmp to parallelize the forloop below
    std::cout << "Max threads available: " << omp_get_max_threads() << std::endl;
    #pragma omp parallel for schedule(dynamic)
    for(int i = 0; i < num_queries; ++i){

        int s, t;
        s = queries[i].first;
        t = queries[i].second;

        // If s == t, path length is 0 but our logic returns k if path > k, so handle separately.
        if (s == t) {
            scores[i] = 0;  // or handle self-loop as needed
            continue;
        }
        
        // BFS from s and t up to given radius
	std::map<int, int> nb_s;// = r_hops[s];
	std::map<int, int> nb_t;// = r_hops[t];
	bfs_limited(adj, nb_s, s, radius);
        bfs_limited(adj, nb_t, t, radius);
        
        // Determine if there is intersection within the neighborhoods
	auto it_s = nb_s.begin();
	auto it_t = nb_t.begin();
	int min_len = 999;
	while (it_s != nb_s.end() && it_t != nb_t.end()){
            if (it_s->first < it_t->first){
                ++it_s;
	    }
	    else if (it_s->first > it_t->first){
                ++it_t;
	    }
	    else{
                int cur_sum = it_s->second + it_t->second;
	       if (cur_sum < min_len) min_len = cur_sum;
	       ++it_s;
	       ++it_t;
	    }
	}
	
	scores[i] = min_len;
    }
    auto end_time = std::chrono::high_resolution_clock::now();
    std::chrono::duration<double> elapsedSeconds = end_time - start_time;

    std::cout << "Elapsed time: " << elapsedSeconds.count() << " seconds" << std::endl;

    
    std::ofstream outfile(output_filename);
    if (!outfile) {
        std::cerr << "Cannot open output file.\n";
        return 1;
    }
    
    for (auto& score : scores) {
        outfile << score << "\n";
    }
    outfile.close();
    
    infile.close();
    return 0;
}
