# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.
# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

"""
Test with:
    python inference_streaming.py --input assets/videos/1.mp4 --output_dir outputs/
"""

import os
import ffmpeg
import numpy as np
import subprocess
import torch  
import tqdm

import videoseal
from videoseal.models import Videoseal
from videoseal.evals.metrics import bit_accuracy

from inference_streaming import detect_video

from scipy.special import betainc

def main(args):
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

    video_model = videoseal.load("videoseal")
    video_model.eval()
    video_model.to(device)
    video_model.compile()

    # Detect the watermark in the video
    soft_msgs = detect_video(video_model, args.input, 16)

    with open(args.key, "r") as f:
        msg = [x == '1' for x in f.readline().strip()]
        
    msgs_ori = torch.tensor(msg).long()[None,...]
    print("msgs_ori = ", msgs_ori)
    bit_acc = bit_accuracy(soft_msgs, msgs_ori).item() * 100
    print(f"Binary message extracted with {bit_acc:.1f}% bit accuracy")
    D = msgs_ori.shape[-1]
    S = bit_acc * D / 100
    #print("correct = %d dim = %d" % (S,D))
    pfa = betainc(S, 1 + (D - S), 0.5) if S else 1
    print(f"pfa = %e" % pfa)

if __name__ == "__main__":
    
    import argparse
    import videoseal.utils as utils

    parser = argparse.ArgumentParser(description="Process a video with Video Seal")
    parser.add_argument("--input", type=str, help="Input video path")
    parser.add_argument("--key", type=str, help="Input key file")
    args = parser.parse_args()

    main(args)
