# Physics-Informed Inductive Biases for Voltage Prediction

This project benchmarks physics-informed machine learning models for voltage prediction in power grid systems. We compare various Graph Neural Network (GNN) architectures with different physics-informed extensions to evaluate their performance on real-world power grid datasets.

## Dependencies

Create and activate a conda environment with all required dependencies:

```bash
# Create conda environment from environment.yml
conda env create -f environment.yml

# Activate the environment
conda activate physics-ml
```

## Data Setup

The project uses the ENGAGE dataset from [Zenodo](https://zenodo.org/records/15464235). 

1. Download the (full) dataset and extract it to `data/ENGAGE_dataset/`
2. The original dataset contains power grid simulation data with voltage measurements
3. We have transformed the data to store slack bus information globally as well as some pandapower network characteristics, creating a graph learning problem where we can focus on PQ buses and compute accurate physics informed losses.

Expected directory structure:
```
data/
└── ENGAGE_dataset/
    ├── 1-LV-rural1--1-no_sw/
    ├── 1-LV-rural2--1-no_sw/
    ├── 1-MV-comm--1-no_sw/
    └── ... (other grid configurations)
```

You will need to run `python generate.py` after this full dataset is extracted, to add the relevant information to the data objects.

## Models

The `models/` folder contains different GNN architectures being compared:

- **Standard GNN**: Baseline graph neural network without physics constraints
- **Residuals GNN**: GNN using skip connections to frame voltage prediction as residuals task
- **Physics-Informed Loss GNN**: GNN with physics-informed loss functions based on AC power flow equations (can be run completed unsupervised, or in addition to standard supervised loss)
- **Complex-Valued GNN**: GNN using complex-valued neural networks to couple voltage magnitude and angle components

All models use PyTorch Geometric for graph-based operations and are designed to predict voltage magnitudes and angles at power grid buses.

## Utils

The `utils/` folder contains key utilities:

- `data_utils.py`: Data loading, preprocessing, and transformation functions
- `training_utils.py`: Training routines, metrics, standard and physics-informed loss functions

## Running Experiments

### Hyperparameter Tuning

We first performed hyperparameter tuning to determine optimal GNN layers and batch sizes:

```bash
python tune.py
```

### Benchmarking

For a quick test run of a single model:

```bash
python run_benchmark.py --data_dir data/ENGAGE_dataset/ --model <MODEL_TYPE> --epochs 5
```

For a full benchmark run of all models:

```bash
time python run_benchmark.py --data_dir data/ENGAGE_dataset/ --model ALL --epochs 3000 --batch_size 128 --save_results --plot
```

The `time` command tracks total experiment duration. Results are automatically saved with timestamps for later analysis.

Available models:
- `gcn-engage`: The GCN model from the original [ENGAGE paper](https://dl.acm.org/doi/10.1145/3679240.3734610)
- `n-gnn`: Standard GNN
- `n-gnn-residuals`: Residuals GNN
- `n-gnn-loss`: Physics-informed GNN, completely unsupervised
- `n-gnn-loss-supervised`: Physics-informed GNN, assisted with standard supervised loss
- `n-gnn-complex`: Complex-Valued GNN
- `ALL`: Run benchmarks for all available models

## Results Analysis

After running experiments, analyze results automatically:

```bash
python analyze_results.py --results_dir out/2025-09-10_01-08-09/
python analyze_model_performance.py out/2025-09-10_01-08-09/results_summary.csv
```

This generates:
- Performance comparison charts (RMSE, MAPE)
- Training time comparisons
- Statistical summaries across all tested grid configurations
- Publication-ready figures saved as high-resolution PNG files
- Tables for summary statistics and full experiment results

## License

This project is licensed under the MIT License.
