import argparse
import matplotlib.pyplot as plt
import pandas as pd
import numpy as np
from pathlib import Path


def get_colors(n_colors):
    """Generate a list of colors for n_colors items using matplotlib's color cycle"""
    if n_colors <= 10:
        # Use matplotlib's default color cycle
        prop_cycle = plt.rcParams['axes.prop_cycle']
        colors = prop_cycle.by_key()['color']
        return (colors * ((n_colors // len(colors)) + 1))[:n_colors]
    else:
        # For many colors, use a colormap
        cmap = plt.cm.tab20  # Good for up to 20 distinct colors
        return [cmap(i / n_colors) for i in range(n_colors)]


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--results_file",
        required=True,
        help="Path to the results CSV file"
    )
    parser.add_argument(
        "--output_dir",
        default=None,
        help="Directory to save the plots (default: 'result_plots' in parent dir of results file)"
    )
    parser.add_argument(
        "--show_plots",
        action="store_true",
        help="Display plots interactively"
    )
    args = parser.parse_args()
    return args


def create_bar_chart(df, testing_grid, metric, output_dir, show_plots=False):
    """Create bar chart for a specific testing grid and metric"""
    
    # Filter data for this testing grid
    grid_data = df[df['testing_grid'] == testing_grid].copy()
    
    if len(grid_data) == 0:
        print(f"Warning: No data found for testing grid '{testing_grid}'")
        return
    
    # Set up the plot
    fig, ax = plt.subplots(figsize=(12, 8))
    
    models = grid_data['model'].tolist()
    x_pos = np.arange(len(models))
    
    if metric == 'rmse':
        # Plot both vm_pu and va_degree for RMSE
        vm_values = grid_data['rmse_vm_pu'].tolist()
        va_values = grid_data['rmse_va_degree'].tolist()
        
        width = 0.35
        bars1 = ax.bar(x_pos - width/2, vm_values, width, label='vm_pu', alpha=0.8)
        bars2 = ax.bar(x_pos + width/2, va_values, width, label='va_degree', alpha=0.8)
        
        ax.set_ylabel('RMSE', fontsize=64, fontweight='bold')
        ax.legend(loc="upper right", fontsize=42)

        # # Add value labels on bars
        # for bar in bars1:
        #     height = bar.get_height()
        #     ax.text(bar.get_x() + bar.get_width()/2., height,
        #            f'{height:.4f}', ha='center', va='bottom', fontsize=9)
        # for bar in bars2:
        #     height = bar.get_height()
        #     ax.text(bar.get_x() + bar.get_width()/2., height,
        #            f'{height:.4f}', ha='center', va='bottom', fontsize=9)
        
    elif metric == 'mape':
        # Plot both vm_pu and va_degree for MAPE
        vm_values = grid_data['mape_vm_pu'].tolist()
        va_values = grid_data['mape_va_degree'].tolist()
        
        width = 0.35
        bars1 = ax.bar(x_pos - width/2, vm_values, width, label='vm_pu', alpha=0.8)
        bars2 = ax.bar(x_pos + width/2, va_values, width, label='va_degree', alpha=0.8)
        
        ax.set_ylabel('MAPE (%)', fontsize=64, fontweight='bold')
        ax.legend(fontsize=42)
        
        # # Add value labels on bars
        # for bar in bars1:
        #     height = bar.get_height()
        #     ax.text(bar.get_x() + bar.get_width()/2., height,
        #            f'{height:.4f}', ha='center', va='bottom', fontsize=9)
        # for bar in bars2:
        #     height = bar.get_height()
        #     ax.text(bar.get_x() + bar.get_width()/2., height,
        #            f'{height:.4f}', ha='center', va='bottom', fontsize=9)
        
    elif metric == 'train_time':
        # Plot only train_time (single metric)
        time_values = grid_data['train_time'].tolist()
        
        bars = ax.bar(x_pos, time_values, alpha=0.8, color='skyblue')
        ax.set_ylabel('Training Time (seconds)', fontsize=64, fontweight='bold')
        
        # # Add value labels on bars
        # for bar in bars:
        #     height = bar.get_height()
        #     ax.text(bar.get_x() + bar.get_width()/2., height,
        #            f'{height:.1f}', ha='center', va='bottom', fontsize=9)
    
    # Common formatting
    ax.set_xlabel('Model', fontsize=64, fontweight='bold')
    ax.set_title(f'{metric.upper()} — {testing_grid}', fontsize=72, fontweight='bold')
    ax.set_xticks(x_pos)
    ax.set_xticklabels(models, rotation=45, ha='right', fontsize=54)
    ax.tick_params(axis='y', labelsize=54)
    ax.grid(True, alpha=0.3, axis='y')
    
    # Adjust layout to prevent label cutoff
    plt.tight_layout()
    
    # Save the plot
    safe_grid_name = testing_grid.replace('/', '_').replace('\\', '_')
    filename = f'{metric}_{safe_grid_name}.png'
    filepath = Path(output_dir) / filename
    plt.savefig(filepath, dpi=300, bbox_inches='tight')
    print(f"Saved plot: {filepath}")
    
    if show_plots:
        plt.show()
    else:
        plt.close()


def create_grouped_bar_chart(df, metric, output_dir, show_plots=False):
    """Create grouped bar chart for a metric across all testing grids"""
    
    # Set up the plot with extra width for many grids
    fig, ax = plt.subplots(figsize=(20, 10))
    
    # Get unique testing grids and models
    testing_grids = sorted(df['testing_grid'].unique())
    models = sorted(df['model'].unique())
    
    x_pos = np.arange(len(testing_grids))
    
    if metric == 'rmse':
        # Plot both vm_pu and va_degree for RMSE
        width = 0.15
        # Generate colors dynamically based on number of model-metric combinations
        n_combinations = len(models) * 2  # 2 metrics per model
        colors = get_colors(n_combinations)
        
        for i, model in enumerate(models):
            vm_values = []
            va_values = []
            
            for grid in testing_grids:
                grid_data = df[(df['testing_grid'] == grid) & (df['model'] == model)]
                if len(grid_data) > 0:
                    vm_values.append(grid_data['rmse_vm_pu'].iloc[0])
                    va_values.append(grid_data['rmse_va_degree'].iloc[0])
                else:
                    vm_values.append(0)
                    va_values.append(0)
            
            # Plot vm_pu and va_degree bars for this model
            bars1 = ax.bar(x_pos + i*width*2 - width/2, vm_values, width, 
                          label=f'{model} (vm_pu)', alpha=0.8, color=colors[i*2])
            bars2 = ax.bar(x_pos + i*width*2 + width/2, va_values, width, 
                          label=f'{model} (va_degree)', alpha=0.6, color=colors[i*2+1])
            
            # Add value labels on bars (only for smaller values to avoid clutter)
            # for bar in bars1:
            #     height = bar.get_height()
            #     if height > 0:
            #         ax.text(bar.get_x() + bar.get_width()/2., height,
            #                f'{height:.3f}', ha='center', va='bottom', fontsize=6, rotation=90)
            # for bar in bars2:
            #     height = bar.get_height()
            #     if height > 0:
            #         ax.text(bar.get_x() + bar.get_width()/2., height,
            #                f'{height:.3f}', ha='center', va='bottom', fontsize=8, rotation=90)
        
        ax.set_ylabel('RMSE', fontsize=64, fontweight='bold')
        
    elif metric == 'mape':
        # Plot both vm_pu and va_degree for MAPE
        width = 0.15
        # Generate colors dynamically based on number of model-metric combinations
        n_combinations = len(models) * 2  # 2 metrics per model
        colors = get_colors(n_combinations)
        
        for i, model in enumerate(models):
            vm_values = []
            va_values = []
            
            for grid in testing_grids:
                grid_data = df[(df['testing_grid'] == grid) & (df['model'] == model)]
                if len(grid_data) > 0:
                    vm_values.append(grid_data['mape_vm_pu'].iloc[0])
                    va_values.append(grid_data['mape_va_degree'].iloc[0])
                else:
                    vm_values.append(0)
                    va_values.append(0)
            
            # Plot vm_pu and va_degree bars for this model
            bars1 = ax.bar(x_pos + i*width*2 - width/2, vm_values, width, 
                          label=f'{model} (vm_pu)', alpha=0.8, color=colors[i*2])
            bars2 = ax.bar(x_pos + i*width*2 + width/2, va_values, width, 
                          label=f'{model} (va_degree)', alpha=0.6, color=colors[i*2+1])
            
            # Add value labels on bars
            # for bar in bars1:
            #     height = bar.get_height()
            #     if height > 0:
            #         ax.text(bar.get_x() + bar.get_width()/2., height,
            #                f'{height:.3f}', ha='center', va='bottom', fontsize=6, rotation=90)
            # for bar in bars2:
            #     height = bar.get_height()
            #     if height > 0:
            #         ax.text(bar.get_x() + bar.get_width()/2., height,
            #                f'{height:.3f}', ha='center', va='bottom', fontsize=8, rotation=90)
        
        ax.set_ylabel('MAPE (%)', fontsize=64, fontweight='bold')
        
    elif metric == 'train_time':
        # Plot only train_time (single metric)
        width = 0.25
        colors = get_colors(len(models))
        
        for i, model in enumerate(models):
            time_values = []
            
            for grid in testing_grids:
                grid_data = df[(df['testing_grid'] == grid) & (df['model'] == model)]
                if len(grid_data) > 0:
                    time_values.append(grid_data['train_time'].iloc[0])
                else:
                    time_values.append(0)
            
            bars = ax.bar(x_pos + i*width - width*(len(models)-1)/2, time_values, width, 
                         label=model, alpha=0.8, color=colors[i])
            
            # Add value labels on bars
            # for bar in bars:
            #     height = bar.get_height()
            #     if height > 0:
            #         ax.text(bar.get_x() + bar.get_width()/2., height,
            #                f'{height:.0f}', ha='center', va='bottom', fontsize=8, rotation=90)
        
        ax.set_ylabel('Training Time (seconds)', fontsize=64, fontweight='bold')
    
    # Common formatting
    ax.set_xlabel('Testing Grid', fontsize=64, fontweight='bold')
    ax.set_title(f'{metric.upper()} — All Experiments Comparison', fontsize=72, fontweight='bold')
    ax.set_xticks(x_pos)
    testing_grid_labels = [grid.split('--')[0][2:] if grid != 'all' else 'All (Known)' for grid in testing_grids]  # Shorten grid names for x-ticks
    ax.set_xticklabels(testing_grid_labels, rotation=45, ha='right', fontsize=54)
    ax.legend(bbox_to_anchor=(1.05, 1), loc='upper left', fontsize=42)
    ax.tick_params(axis='y', labelsize=54)
    ax.grid(True, alpha=0.3, axis='y')
    
    # Adjust layout to prevent label cutoff
    plt.tight_layout()
    
    # Save the plot
    filename = f'{metric}_all_grids_comparison.png'
    filepath = Path(output_dir) / filename
    plt.savefig(filepath, dpi=300, bbox_inches='tight')
    print(f"Saved grouped plot: {filepath}")
    
    if show_plots:
        plt.show()
    else:
        plt.close()


def create_metric_attribute_bar_chart(df, metric, attribute, output_dir, show_plots=False):
    """Create bar chart for a specific (metric, attribute) combination across all testing grids"""
    
    # Set up the plot with extra width for many grids
    fig, ax = plt.subplots(figsize=(20, 15))
    
    # Get unique testing grids and models
    testing_grids = sorted(df['testing_grid'].unique())
    models = sorted(df['model'].unique())
    
    x_pos = np.arange(len(testing_grids))
    
    # Define the specific metric column based on metric and attribute
    if metric == 'rmse' and attribute == 'vm_pu':
        metric_column = 'rmse_vm_pu'
        ylabel = 'RMSE (pu)'
        title = 'RMSE — Voltage Magnitude'
    elif metric == 'rmse' and attribute == 'va_degree':
        metric_column = 'rmse_va_degree'
        ylabel = 'RMSE (degree)'
        title = 'RMSE — Voltage Angle'
    elif metric == 'mape' and attribute == 'vm_pu':
        metric_column = 'mape_vm_pu'
        ylabel = 'MAPE (pu)'
        title = 'MAPE — Voltage Magnitude'
    elif metric == 'mape' and attribute == 'va_degree':
        metric_column = 'mape_va_degree'
        ylabel = 'MAPE (degree)'
        title = 'MAPE — Voltage Angle'
    elif metric == 'train_time':
        metric_column = 'train_time'
        ylabel = 'Training Time (s)'
        title = 'Training Time'
    else:
        print(f"Warning: Unknown metric-attribute combination '{metric}'-'{attribute}'")
        return
    
    # Generate colors dynamically based on number of models
    colors = get_colors(len(models))
    
    # Adjust width and spacing based on number of models to maintain consistent gaps
    n_models = len(models)
    if n_models <= 2:
        width = 0.45
    elif n_models == 3:
        width = 0.3
    elif n_models == 4:
        width = 0.23
    else:
        # For 5+ models, reduce width further
        width = max(0.12, 0.9 / (n_models + 0.5))  # Ensure minimum width
    
    for i, model in enumerate(models):
        values = []
        
        for grid in testing_grids:
            grid_data = df[(df['testing_grid'] == grid) & (df['model'] == model)]
            if len(grid_data) > 0:
                values.append(grid_data[metric_column].iloc[0])
            else:
                values.append(0)
        
        # Calculate bar position with consistent spacing
        # Center the group and space bars evenly within the group
        # start_offset = -group_width / 2
        # bar_spacing = group_width / n_models if n_models > 1 else 0
        # bar_position = x_pos + start_offset + (i + 0.5) * bar_spacing

        bar_position = x_pos + i*width - width*(len(models)-1)/2
        
        # Create bars
        model_label = {
            'NormedGNN': 'Base',
            'NormedGNN_PhysicsLoss': 'Phys-Loss',
            'NormedGNN_PhysicsLoss_Supervised': 'Phys-Loss',
            'NormedGNN_Residuals': 'Residuals',
            'NormedGNN_Complex': 'Complex',
        }
        bars = ax.bar(bar_position, values, width, 
                     label=model_label.get(model, model), 
                     alpha=0.8, 
                     color=colors[i])
        
        # # Add value labels on bars
        # for bar in bars:
        #     height = bar.get_height()
        #     if height > 0:
        #         # Format based on metric type
        #         if metric == 'train_time':
        #             label_text = f'{height:.0f}'
        #         else:
        #             label_text = f'{height:.4f}'
                
        #         ax.text(bar.get_x() + bar.get_width()/2., height,
        #                label_text, ha='center', va='bottom', 
        #                fontsize=8, rotation=90)
    
    # Common formatting
    ax.set_xlabel('Testing Grid', fontsize=64, fontweight='bold')
    ax.set_ylabel(ylabel, fontsize=64, fontweight='bold')
    ax.set_title(title, fontsize=72, fontweight='bold')
    ax.set_xticks(x_pos)
    testing_grid_labels = [grid.split('--')[0][2:] if grid != 'all' else 'All (Known)' for grid in testing_grids]  # Shorten grid names for x-ticks
    ax.set_xticklabels(testing_grid_labels, rotation=45, ha='right', fontsize=54)
    # ax.legend(bbox_to_anchor=(1.05, 1), loc='upper left', fontsize=54)
    ax.legend(loc='upper right', fontsize=42)
    ax.tick_params(axis='y', labelsize=54)
    ax.grid(True, alpha=0.3, axis='y')
    ax.margins(x=0.02)
    
    # Adjust layout to prevent label cutoff
    plt.tight_layout()
    
    # Save the plot
    if metric == 'train_time':
        filename = f'{metric}_all_grids_comparison.png'
    else:
        filename = f'{metric}_{attribute}_all_grids_comparison.png'
    filepath = Path(output_dir) / filename
    plt.savefig(filepath, dpi=300, bbox_inches='tight')
    print(f"Saved metric-attribute plot: {filepath}")
    
    if show_plots:
        plt.show()
    else:
        plt.close()


def create_metric_attribute_box_plot(df, metric, attribute, output_dir, show_plots=False):
    """Create box plot showing performance distribution of each model across all testing grids"""
    
    # Set up the plot
    fig, ax = plt.subplots(figsize=(20, 15))
    
    # Get unique models
    models = sorted(df['model'].unique())
    
    # Define the specific metric column based on metric and attribute
    if metric == 'rmse' and attribute == 'vm_pu':
        metric_column = 'rmse_vm_pu'
        ylabel = 'RMSE (pu)'
        title = 'RMSE Distribution — Voltage Magnitude'
    elif metric == 'rmse' and attribute == 'va_degree':
        metric_column = 'rmse_va_degree'
        ylabel = 'RMSE (degree)'
        title = 'RMSE Distribution — Voltage Angle'
    elif metric == 'mape' and attribute == 'vm_pu':
        metric_column = 'mape_vm_pu'
        ylabel = 'MAPE (% pu)'
        title = 'MAPE Distribution — Voltage Magnitude'
    elif metric == 'mape' and attribute == 'va_degree':
        metric_column = 'mape_va_degree'
        ylabel = 'MAPE (% degree)'
        title = 'MAPE Distribution — Voltage Angle'
    elif metric == 'train_time':
        metric_column = 'train_time'
        ylabel = 'Training Time (s)'
        title = 'Training Time Distribution'
    else:
        print(f"Warning: Unknown metric-attribute combination '{metric}'-'{attribute}'")
        return
    
    # Collect data for each model
    box_data = []
    box_labels = []

    model_label = {
        'NormedGNN': 'Base',
        'NormedGNN_PhysicsLoss': 'Phys-Loss',
        'NormedGNN_PhysicsLoss_Supervised': 'Phys-Loss',
        'NormedGNN_Residuals': 'Residuals',
        'NormedGNN_Complex': 'Complex',
    }
    for model in models:
        model_data = df[df['model'] == model][metric_column].tolist()
        if len(model_data) > 0:
            box_data.append(model_data)
            box_labels.append(model_label.get(model, model))
    
    # Create box plot
    box_plot = ax.boxplot(box_data, patch_artist=True)
    
    # Set x-axis labels
    ax.set_xticklabels(box_labels, rotation=45, ha='right', fontsize=54)
    
    # Color the boxes dynamically
    colors = get_colors(len(box_data))
    for patch, color in zip(box_plot['boxes'], colors):
        patch.set_facecolor(color)
        patch.set_alpha(0.7)
    
    # Add individual data points as scatter
    for i, (model_data, model) in enumerate(zip(box_data, box_labels)):
        x = np.random.normal(i + 1, 0.04, size=len(model_data))  # Add slight jitter
        ax.scatter(x, model_data, alpha=0.6, color='black', s=20, zorder=3)
    
    # Formatting
    ax.set_xlabel('Model', fontsize=64, fontweight='bold')
    ax.set_ylabel(ylabel, fontsize=64, fontweight='bold')
    ax.set_title(title, fontsize=72, fontweight='bold')
    ax.tick_params(axis='x', labelsize=54)
    ax.tick_params(axis='y', labelsize=54)
    ax.grid(True, alpha=0.3, axis='y')
    
    # Add statistics text
    stats_text = []
    for i, (model_data, model) in enumerate(zip(box_data, box_labels)):
        mean_val = np.mean(model_data)
        std_val = np.std(model_data)
        if metric == 'train_time':
            stats_text.append(f'{model}: μ={mean_val:.1f}, σ={std_val:.1f}')
        else:
            stats_text.append(f'{model}: μ={mean_val:.4f}, σ={std_val:.4f}')
    
    # Add text box with statistics
    # stats_str = '\n'.join(stats_text)
    # ax.text(0.02, 0.98, stats_str, transform=ax.transAxes, fontsize=36,
    #         verticalalignment='top', bbox=dict(boxstyle='round', facecolor='wheat', alpha=0.8))
    
    plt.tight_layout()
    
    # Save the plot
    if metric == 'train_time':
        filename = f'{metric}_boxplot_distribution.png'
    else:
        filename = f'{metric}_{attribute}_boxplot_distribution.png'
    filepath = Path(output_dir) / filename
    plt.savefig(filepath, dpi=300, bbox_inches='tight')
    print(f"Saved box plot: {filepath}")
    
    if show_plots:
        plt.show()
    else:
        plt.close()


def create_comprehensive_attribute_plots(df, output_dir, show_plots=False):
    """Create comprehensive plots split by (metric, attribute) pairs - resulting in 5 bar charts + 5 box plots"""
    
    print("\nCreating comprehensive metric-attribute based plots...")
    
    # Create plots for each (metric, attribute) combination
    metric_attribute_pairs = [
        ('rmse', 'vm_pu'),
        ('rmse', 'va_degree'),
        ('mape', 'vm_pu'),
        ('mape', 'va_degree'),
        ('train_time', None)  # Training time doesn't have an attribute distinction
    ]
    
    # Create bar charts
    print("Creating bar charts...")
    for metric, attribute in metric_attribute_pairs:
        if metric == 'train_time':
            print(f"Creating comprehensive bar chart for {metric}")
        else:
            print(f"Creating comprehensive bar chart for {metric} - {attribute}")
        create_metric_attribute_bar_chart(df, metric, attribute, output_dir, show_plots)
    
    # Create box plots
    print("Creating box plots...")
    for metric, attribute in metric_attribute_pairs:
        if metric == 'train_time':
            print(f"Creating box plot for {metric}")
        else:
            print(f"Creating box plot for {metric} - {attribute}")
        create_metric_attribute_box_plot(df, metric, attribute, output_dir, show_plots)


def analyze_results(results_file, output_dir=None, show_plots=False):
    """Main function to analyze results and create plots"""
    
    # Set default output directory if not provided
    if output_dir is None:
        results_path = Path(results_file)
        output_dir = results_path.parent / "result_plots"
    else:
        output_dir = Path(output_dir)
    
    # Create organized subdirectories
    individual_plots_dir = output_dir / "individual_grid_analysis"
    metric_plots_dir = output_dir / "metric_based_comparison"
    metric_attribute_plots_dir = output_dir / "metric_attribute_comparison"
    
    # Create all directories
    individual_plots_dir.mkdir(parents=True, exist_ok=True)
    metric_plots_dir.mkdir(parents=True, exist_ok=True)
    metric_attribute_plots_dir.mkdir(parents=True, exist_ok=True)
    
    print("Creating organized plot directories:")
    print(f"  Individual plots: {individual_plots_dir}")
    print(f"  Metric-based plots: {metric_plots_dir}")
    print(f"  Metric-attribute plots: {metric_attribute_plots_dir}")
    
    # Read results
    try:
        df = pd.read_csv(results_file)
    except FileNotFoundError:
        print(f"Error: Results file '{results_file}' not found")
        return
    except Exception as e:
        print(f"Error reading results file: {e}")
        return
    
    # Check required columns
    required_columns = ['model', 'testing_grid', 'rmse_vm_pu', 'rmse_va_degree', 
                       'mape_vm_pu', 'mape_va_degree', 'train_time']
    missing_columns = [col for col in required_columns if col not in df.columns]
    if missing_columns:
        print(f"Error: Missing required columns: {missing_columns}")
        return
    
    # Get unique testing grids
    testing_grids = df['testing_grid'].unique()
    print(f"Found {len(testing_grids)} testing grids: {list(testing_grids)}")
    
    # Create individual plots for each testing grid and metric
    print("\n" + "="*60)
    print("CREATING INDIVIDUAL GRID ANALYSIS PLOTS")
    print("="*60)
    metrics = ['rmse', 'mape', 'train_time']
    
    total_plots = len(testing_grids) * len(metrics)
    current_plot = 0
    
    # for testing_grid in testing_grids:
    #     for metric in metrics:
    #         current_plot += 1
    #         print(f"Creating individual plot {current_plot}/{total_plots}: {metric} for {testing_grid}")
    #         create_bar_chart(df, testing_grid, metric, individual_plots_dir, show_plots)
    
    # Create grouped plots by metric
    print("\n" + "="*60)
    print("CREATING METRIC-BASED COMPARISON PLOTS")
    print("="*60)
    for metric in metrics:
        print(f"Creating metric-based plot for {metric}")
        create_grouped_bar_chart(df, metric, metric_plots_dir, show_plots)
    
    # Create comprehensive metric-attribute plots
    print("\n" + "="*60)
    print("CREATING METRIC-ATTRIBUTE COMPARISON PLOTS")
    print("="*60)
    create_comprehensive_attribute_plots(df, metric_attribute_plots_dir, show_plots)
    
    # Print summary statistics
    print("\n" + "="*60)
    print("SUMMARY STATISTICS")
    print("="*60)
    
    for testing_grid in testing_grids:
        grid_data = df[df['testing_grid'] == testing_grid]
        print(f"\nTesting Grid: {testing_grid}")
        print("-" * 40)
        
        for _, row in grid_data.iterrows():
            print(f"Model: {row['model']}")
            print(f"  RMSE (vm_pu): {row['rmse_vm_pu']:.6f}")
            print(f"  RMSE (va_degree): {row['rmse_va_degree']:.6f}")
            print(f"  MAPE (vm_pu): {row['mape_vm_pu']:.6f}")
            print(f"  MAPE (va_degree): {row['mape_va_degree']:.6f}")
            print(f"  Training Time: {row['train_time']:.1f}s")
            print()

    print(f"\nAll plots saved to: {output_dir}")


def main():
    args = parse_args()
    analyze_results(args.results_file, args.output_dir, args.show_plots)


if __name__ == '__main__':
    main()
