import numpy as np
from typing import Optional, Sequence, Tuple, List

# Minimal, reusable visualization helpers for grid-based duel-like games.
# Assumptions:
# - Board is HxW grid.
# - Obstacles mask provided as boolean array (H,W).
# - Player positions provided as tuples (r,c) or list of tuples per side.
# - Action encoding: 0..3 moves (U,D,L,R), 4..7 shoots (U,D,L,R).
# - Optional safe-zone bounds (min_r, min_c, max_r, max_c); cells outside are marked unsafe.

try:
    import matplotlib.pyplot as plt
    from matplotlib.patches import Patch, Rectangle, FancyArrowPatch
    from matplotlib.lines import Line2D
except Exception:  # pragma: no cover
    plt = None
    Patch = Rectangle = FancyArrowPatch = Line2D = None

MoveDirs: List[Tuple[int,int]] = [(-1,0),(1,0),(0,-1),(0,1)]


def make_grid(H:int, W:int,
              obstacles: np.ndarray,
              unsafe_bounds: Optional[Tuple[int,int,int,int]],
              p1_positions: Sequence[Tuple[int,int]] | Optional[Tuple[int,int]],
              p2_positions: Sequence[Tuple[int,int]] | Optional[Tuple[int,int]],
              p1_base_color=(0.2, 0.4, 0.9),
              p2_base_color=(0.9, 0.3, 0.2),
              unsafe_color=(0.85, 0.85, 0.85)) -> np.ndarray:
    """Build an RGB grid image for imshow.

    p1_positions/p2_positions can be single tuple or list of tuples (some may be None).
    obstacles is boolean mask (H,W) where True marks an obstacle.
    """
    grid = np.ones((H, W, 3), dtype=float)
    # Unsafe area outside bounds
    if unsafe_bounds is not None:
        min_r, min_c, max_r, max_c = unsafe_bounds
        for r in range(H):
            for c in range(W):
                if not (min_r <= r <= max_r and min_c <= c <= max_c):
                    grid[r, c] = unsafe_color
    # Obstacles black
    grid[obstacles] = (0, 0, 0)
    # Normalize inputs to lists
    def to_list(pos):
        if pos is None:
            return []
        # If it's a sequence of positions (possibly containing None), detect by inspecting any element
        if isinstance(pos, (list, tuple)):
            if any(isinstance(p, (list, tuple, np.ndarray)) for p in pos):
                return [p for p in pos if p is not None]
            # Single position tuple like (r,c)
            if len(pos) == 2 and all(isinstance(x, (int, np.integer)) for x in pos):
                return [pos]
        return [pos] if pos is not None else []
    def norm_pos(p) -> Optional[Tuple[int,int]]:
        try:
            if isinstance(p, np.ndarray):
                if p.ndim == 1 and p.size == 2:
                    return (int(p[0]), int(p[1]))
                # Some code might pass arrays of shape (2,) or lists of numpy scalars
                if p.shape == (2,):
                    return (int(p[0]), int(p[1]))
            if isinstance(p, (list, tuple)) and len(p) == 2:
                return (int(p[0]), int(p[1]))
        except Exception:
            return None
        return None
    for i, p in enumerate(to_list(p1_positions)):
        q = norm_pos(p)
        if q is None:
            continue
        r, c = q
        if 0 <= r < H and 0 <= c < W:
            grid[r, c] = (p1_base_color[0], min(1.0, p1_base_color[1] + 0.1 * i), p1_base_color[2])
    for i, p in enumerate(to_list(p2_positions)):
        q = norm_pos(p)
        if q is None:
            continue
        r, c = q
        if 0 <= r < H and 0 <= c < W:
            grid[r, c] = (p2_base_color[0], min(1.0, p2_base_color[1] + 0.1 * i), p2_base_color[2])
    return grid


def draw_board(ax, grid: np.ndarray, H:int, W:int,
               safe_bounds: Optional[Tuple[int,int,int,int]] = None,
               center_marker: Optional[Tuple[int,int]] = None) -> None:
    im = ax.imshow(grid, origin='upper')
    ax.set_xticks(np.arange(-.5, W, 1), minor=True)
    ax.set_yticks(np.arange(-.5, H, 1), minor=True)
    ax.grid(which='minor', color='gray', linewidth=0.5, alpha=0.5)
    ax.tick_params(left=False, bottom=False, labelleft=False, labelbottom=False)
    if safe_bounds is not None:
        min_r, min_c, max_r, max_c = safe_bounds
        rect = Rectangle((min_c - 0.5, min_r - 0.5), (max_c - min_c + 1), (max_r - min_r + 1),
                         fill=False, edgecolor=(0.1, 0.7, 0.1), linewidth=1.5, linestyle='--', alpha=0.9)
        ax.add_patch(rect)
    if center_marker is not None:
        ax.plot([center_marker[1]], [center_marker[0]], marker='x', color='yellow', markersize=10, mew=2, zorder=4)


def _shoot_endpoint(r0:int, c0:int, dr:int, dc:int, H:int, W:int,
                    shoot_range:int,
                    obstacles: np.ndarray,
                    safe_bounds: Optional[Tuple[int,int,int,int]]) -> Tuple[int,int]:
    r1, c1 = r0, c0
    for dist in range(1, shoot_range + 1):
        tr, tc = r0 + dr*dist, c0 + dc*dist
        if not (0 <= tr < H and 0 <= tc < W):
            break
        if obstacles[tr, tc]:
            break
        if safe_bounds is not None:
            min_r, min_c, max_r, max_c = safe_bounds
            if not (min_r <= tr <= max_r and min_c <= tc <= max_c):
                break
        r1, c1 = tr, tc
    return r1, c1


def draw_action_arrow(ax,
                      action:int,
                      actor_color: Tuple[float,float,float],
                      start_pos: Tuple[int,int],
                      H:int, W:int,
                      shoot_range:int,
                      obstacles: np.ndarray,
                      safe_bounds: Optional[Tuple[int,int,int,int]]) -> None:
    r0, c0 = start_pos
    if 0 <= action <= 3:
        dr, dc = MoveDirs[action]
        r1, c1 = r0 + dr, c0 + dc
        r1 = max(0, min(H-1, r1)); c1 = max(0, min(W-1, c1))
        is_shot = False
    else:
        dr, dc = MoveDirs[action - 4]
        r1, c1 = _shoot_endpoint(r0, c0, dr, dc, H, W, shoot_range, obstacles, safe_bounds)
        is_shot = True
    style = '-|>' if not is_shot else '->'
    ls = 'solid' if not is_shot else 'dashed'
    lw = 2.0 if not is_shot else 2.2
    arrow = FancyArrowPatch((c0, r0), (c1, r1), arrowstyle=style,
                             mutation_scale=12, linewidth=lw, linestyle=ls,
                             color=actor_color, alpha=0.95, zorder=5)
    ax.add_patch(arrow)


def draw_joint_action_arrows(ax,
                             actions: Sequence[int],
                             team_positions: Sequence[Optional[Tuple[int,int]]],
                             actor_color: Tuple[float,float,float],
                             H:int, W:int,
                             shoot_range:int,
                             obstacles: np.ndarray,
                             safe_bounds: Optional[Tuple[int,int,int,int]]):
    for pos, a in zip(team_positions, actions):
        if pos is None:
            continue
        draw_action_arrow(ax, int(a), actor_color, pos, H, W, shoot_range, obstacles, safe_bounds)


def add_legend(ax, legend_outside: bool=True) -> None:
    legend = [
        Patch(facecolor=(1,1,1), edgecolor='gray', label='Empty'),
        Patch(facecolor=(0.85,0.85,0.85), edgecolor='gray', label='Unsafe'),
        Patch(facecolor=(0,0,0), edgecolor='gray', label='Obstacle'),
        Patch(facecolor=(0.2,0.4,0.9), edgecolor='gray', label='P1'),
        Patch(facecolor=(0.9,0.3,0.2), edgecolor='gray', label='P2'),
        Line2D([0],[0], color='k', lw=2.0, linestyle='solid', label='Move'),
        Line2D([0],[0], color='k', lw=2.2, linestyle='dashed', label='Shoot'),
    ]
    if legend_outside:
        ax.legend(handles=legend, loc='upper left', bbox_to_anchor=(1.02, 1.0), frameon=False, fontsize=8)
    else:
        ax.legend(handles=legend, loc='upper right', frameon=False, fontsize=8)


__all__ = [
    'make_grid',
    'draw_board',
    'draw_action_arrow',
    'draw_joint_action_arrows',
    'add_legend',
]
