# Synthesizing Character-Scene Pool Pipeline

An end-to-end synthesis pipeline for generating role-playing scenarios that automatically creates worldviews, synthetic character profiles, scene settings, and initial dialogues at any scale.

## 🌟 Overview

This synthesis pipeline generates high-quality role-playing scenarios through five core steps:

1. **Topic Generation** - Create original novel worldview themes
2. **Worldview Construction** - Generate detailed worldview settings based on themes
3. **Character Generation** - Create character profiles that fit the worldview
4. **Scene Generation** - Design specific character interaction scenarios
5. **Dialogue Generation** - Produce initial character dialogues

## 📁 Project Structure

```
Synthesizing/
├── README.md                                    # This document
├── synthesizing_character_scene_pool_pipeline.py # Main pipeline script
├── utils.py                                     # Common utility functions
├── prompt_template/                             # Prompt templates
│   ├── dataset.json                            # Dataset template
│   ├── dialogue.json                           # Dialogue template
│   ├── initializing_topics.json                # Initial topic template
│   ├── prompt.py                               # Prompt template collection
│   └── scoring_topics.json                     # Topic scoring template
├── frontend_preview/                           # Web preview interface
│   ├── app.py                                  # Flask application
│   └── templates/
│       └── preview.html                        # Preview page
├── cache/                                      # Cache directory
│   ├── best_topic/                            # Best topics
│   ├── character_profile/                     # Character profiles
│   ├── raw_topics/                            # Raw topics
│   ├── scenes/                                # Scene files
│   ├── scored_topics/                         # Scored topics
│   ├── worldview/                             # Worldview files
│   └── tracking.json.example                  # Tracking file example
├── synthetic_character_scene_pool/             # Output result
└── Module scripts:
    ├── topic_generation.py                    # Topic generation
    ├── worldview_generation.py                # Worldview generation
    ├── character_generation.py                # Character generation
    ├── scene_generation.py                    # Scene generation
    ├── initial_dialogue_generation.py         # Initial dialogue generation
    └── character_selector.py                  # Character selector
```

## 🚀 Quick Start

### Environment Setup

1. **Install Dependencies**:
```bash
pip install -r requirements.txt
```

2. **Configure API Keys**:
Create a `.env` file and add the corresponding API keys, the more model you can use, the more diverse of the sampled topics:
```env
OPENAI_API_KEY=
OPENAI_BASE_URL=

ANTHROPIC_API_KEY=
ANTHROPIC_BASE_URL=

GEMINI_API_KEY=
GEMINI_BASE_URL=

DEEPSEEK_API_KEY=
DEEPSEEK_BASE_URL=

QWEN_API_KEY=
QWEN_BASE_URL=
```

### Basic Usage

#### Run Complete Pipeline from scratch

```bash
python synthesizing_character_scene_pool_pipeline.py
```

#### Custom Parameters

Specify available model and parameters:
```bash
python synthesizing_character_scene_pool_pipeline.py \
    --model claude-3-7-sonnet-20250219 \
    --temperature 1.0 \
    --num-characters 3
```

Enable selector by activate interactive frontend preview:
```bash
python synthesizing_character_scene_pool_pipeline.py --use-selector
```

An example of the Synthesized Character-Scene Pool Preview is shown below.

![](../Figs/synthesized character-scene pool preview.png)

You can also get direct access to the frontend interface for:

- Previewing generated characters and scenes
- Browsing historical generation results
- Managing character files

Start the preview server:

```bash
cd frontend_preview
python app.py
```

Then visit `http://localhost:1210` to view the interface.



## 📝 Pipeline Description

![](../Figs/pipeline_new.png)

The entire pipeline is mainly based on LLM. Several mainstream LLMs are first employed to sample diverse thematic seeds, from which a coherent fictional worldview is constructed. Based on this worldview, multiple original character profiles are generated. A narrative scene is then created involving the selected characters, culminating in a dialogue that reflects their roles and relationships. Each intermediate step undergoes iterative self-review and refinement to ensure logical consistency, narrative coherence, and linguistic fluency.



## 📊 Output File Description

### YAML Format Files

- **Worldview Files**: Structured information containing world settings
- **Character Profiles**: Detailed character attributes and backgrounds
- **Scene Files**: Environment and situational descriptions of scenes

### JSON Format Files

- **Dialogue Files**: Dialogue content between characters, including speakers, content, emotions, etc.
- **Theme Files**: Generated and scored theme information
- **Tracking Files**: Record associations between files

### Directory Structure Description

Generated files are organized by timestamp for easy version management and tracking:

```
cache/
├── worldview/
│   └── worldview_20250924_143022.yaml
├── character_profile/
│   └── character_profile_20250924_143022.yaml
├── synthetic_character_scene_pool/
│   └── selected_scene_20250924_143022/
│       ├── character_1.yaml
│       ├── character_2.yaml
│		├── ...
│       └── character_n.yaml
├── scene/
│   ├── scene_20250924_143022.yaml
└── dialogue/
    └── dialogue_20250924_143022.json
```

## 🔧 Customization and Extension

### Modify Prompt Templates

All AI prompt templates are in the `prompt_template/` directory and can be modified as needed:

- `initializing_topics.json` - Topic generation template
- `scoring_topics.json` - Topic scoring template
- `dialogue.json` - Dialogue generation template
- `dataset.json` - Dataset template
- `prompt.py` - Python file containing various prompt templates

### Add New Large Language Models

Add new model support in `utils.py`:

1. Implement the model's API call function
2. Add model judgment logic in the `get_llm_response()` function
3. Update the list of supported models

## 🚨 Important Notes

1. **API Limitations**: Please note the API call limitations and costs of various LLM services
2. **Generation Quality**: Different models may have varying generation quality; testing is recommended to select the most suitable model
3. **File Management**: Large-scale generation will produce many files; regular cleanup of unnecessary results is recommended
4. **Network Connection**: Ensure stable network connection; the generation process may take considerable time

## 📄 License

Please refer to the license file in the project root directory.

---

**Enjoy your role-playing scenario creation journey!** 🎭✨