import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.vdn import VDNMixer
from modules.mixers.qmix import QMixer
from modules.mixers.cw_qmix import CW_QMixer
import torch as th
from torch.optim import RMSprop
from torch.distributions import Categorical
from torch.distributions import Beta
from torch.distributions import Dirichlet


class QLearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger

        self.params = list(mac.parameters())

        self.last_target_update_episode = 0

        self.mixer = None
        if args.mixer is not None:
            if args.mixer == "vdn":
                self.mixer = VDNMixer()
            elif args.mixer == "qmix":
                self.mixer = QMixer(args)
            else:
                raise ValueError("Mixer {} not recognised.".format(args.mixer))

            self.mixer = CW_QMixer(args)
            self.params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)

        self.optimiser = RMSprop(params=self.params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        self.log_stats_t = -self.args.learner_log_interval - 1

    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int):
        # Get the relevant quantities
        rewards = batch["reward"][:, :-1]
        actions = batch["actions"][:, :-1]
        terminated = batch["terminated"][:, :-1].float()
        mask = batch["filled"][:, :-1].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]

        # print (batch["obs"].size())
        # Calculate estimated Q-Values
        mac_out = []
        mac_out2 = []
        self.mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            agent_outs = self.mac.forward(batch, t=t)
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time


        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

        # Calculate the Q-Values necessary for the target
        target_mac_out = []

        self.target_mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            target_agent_outs = self.target_mac.forward(batch, t=t)
            target_mac_out.append(target_agent_outs)

        target_mac_out_detach = th.stack(target_mac_out, dim=1).clone().detach()

        # We don't need the first timesteps Q-Value estimate for calculating targets
        target_mac_out = th.stack(target_mac_out[1:], dim=1)  # Concat across time

        # Mask out unavailable actions
        target_mac_out[avail_actions[:, 1:] == 0] = -9999999


        mac_out_detach = mac_out.clone().detach()
        mac_out_detach[avail_actions == 0] = -9999999
        cur_max_actions = mac_out_detach.max(dim=3, keepdim=True)[1]


        target_mac_out_detach[avail_actions == 0] = -9999999
        target_max_actions = target_mac_out_detach.max(dim=3, keepdim=True)[1]


        # Max over target Q-Values
        if self.args.double_q:
            # Get actions that maximise live Q (for double q-learning)
            target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions[:, 1:]).squeeze(3)
        else:
            target_max_qvals = target_mac_out.max(dim=3)[0]

        max_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=cur_max_actions[:, :-1]).squeeze(3)


        # chosen_action_qvals3 = chosen_action_qvals

        # Mix
        if self.mixer is not None:
            chosen_action_qvals, chosen_action_qvals2 = self.mixer(chosen_action_qvals, batch["state"][:, :-1])
            max_chosen_action_qvals, max_chosen_action_qvals2 = self.mixer(max_action_qvals, batch["state"][:, :-1])
            target_max_qvals, target_max_qvals2 = self.target_mixer(target_max_qvals, batch["state"][:, 1:])

        # v17

        alpha = 0.75
        targets = (rewards + self.args.gamma * (1 - terminated) * target_max_qvals).detach()

        # Td-error
        td_error = (chosen_action_qvals - targets)
        td_error2 = (chosen_action_qvals2 - targets)

        cond_1 = targets
        cond_2 = max_chosen_action_qvals

        weight = th.where(cond_1 > cond_2, targets * 0 + 1, targets * 0  + alpha)
        weight = th.where(chosen_action_qvals == max_chosen_action_qvals, targets * 0 + 1, weight)

        size = targets.size()
        debug_pos = th.where(cond_1 > cond_2, targets * 0 + 1, targets * 0 + 0)
        debug_neg = th.where(cond_1 > cond_2, targets * 0 + 0, targets * 0 + 1)

        mask = mask.expand_as(td_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask
        masked_td_error2 = td_error2 * mask
        masked_debug_pos = debug_pos * mask
        masked_debug_neg = debug_neg * mask

        pos_ratio = masked_debug_pos.sum() / (masked_debug_pos.sum() + masked_debug_neg.sum())


        # Normal L2 loss, take mean over actual data
        loss = 1 * (masked_td_error ** 2).sum() / mask.sum()
        loss += (weight * (masked_td_error2 ** 2)).sum() / mask.sum()


        # Optimise
        self.optimiser.zero_grad()
        loss.backward()
        grad_norm = th.nn.utils.clip_grad_norm_(self.params, self.args.grad_norm_clip)
        self.optimiser.step()

        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("loss", loss.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("td_error_abs2", (masked_td_error2.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("q_taken_mean", (chosen_action_qvals * mask).sum().item()/(mask_elems), t_env)
            self.logger.log_stat("q_taken_mean2", (chosen_action_qvals2 * mask).sum().item()/(mask_elems), t_env)
            self.logger.log_stat("q_taken_mean5", (max_chosen_action_qvals * mask).sum().item()/(mask_elems), t_env)
            self.logger.log_stat("q_taken_mean6", (max_chosen_action_qvals2 * mask).sum().item()/(mask_elems), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item()/(mask_elems), t_env)
            self.logger.log_stat("pos_ratio", pos_ratio.item(), t_env)



            self.log_stats_t = t_env

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()

    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        self.optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))
