########################################################
# This function file contains four functions that are used in the simulation
# 1. The function **sqrt_Sigma** creates the sigma matrix with decay and rho parameters.
# 2. The function **sqrt_Sigma1** creates a low-rank sigma matrix.
# 3. The function **sqrt_Sigma2** creates a sigma matrix with fast decaying rate of eigenvalues.
# 4. The function **sqrt_Sigma3** creates a sigma matrix, with the smallest r-dimensional subspace containing the true beta.
########################################################

 # Input: p is the dimension, rho_0 and decay are parameters to control the correlations;
 # Output: the square of Covariance matrix
sqrt_Sigma=function(p,rho_0,decay,type) 
{
  if (type==0)
{
  Sigma=diag(rep(1,p))
  for (i in 2:p)          # last eigenvalue is 0
  { 
    for (j in 1:(i-1))
    {
    Sigma[i,j]=rho_0^(abs(i-j)/decay)
    Sigma[j,i]=Sigma[i,j]
    }
  }
  E=eigen(Sigma)
  V=E$vectors
  D=E$values
  D[D<0]=min(D[D>0])   
  sqrt_Sigma=V%*%diag(D^(1/2))%*%t(V)  # root covariance 
}
  if (type==1)  # sigma is the identity matrix
{
  sqrt_Sigma=diag(p)
}
  if (type==2) # sigma is a low-rank matrix
{ 
  r = 3
  U = matrix(rnorm(p*r),nrow=p)
  Sigma = U%*%t(U)/3
  E=eigen(Sigma)
  V=E$vectors
  D=E$values
  D=D[D>0.00001]
  sqrt_Sigma=V[,1:length(D)]%*%diag(D^(1/2))%*%t(V[,1:length(D)])
}
 return(sqrt_Sigma)
}

######################################
### generate square of a low-rank covariance matrix. Input: rk is the rank and p is the dimension.
sqrt_Sigma1=function(rk,p)
{
  rd=matrix(rnorm(p*p),ncol=p)
  V=eigen(rd%*%t(rd))
  eigvalue_lambda= diag(c(rep(p^(1/2),rk),1/c(1:(p-rk))))  #rep(0,p-rk))
  sqrt_Sigma=V$vectors%*%eigvalue_lambda%*%t(V$vectors) 
  return(sqrt_Sigma)
}

###################################
# Input: p is the dimension of predictors. Output: the square of the covariance matrix
sqrt_Sigma2=function(p)
{
  rd = matrix(rnorm(p*p),ncol=p)
  V = eigen(rd%*%t(rd))
  eigvalue_lambda = exp(-0.5*c(p:1))
  eigvalue_lambda = p*eigvalue_lambda/(sum(eigvalue_lambda))
  sqrt_Sigma = V$vectors%*%diag(sqrt(eigvalue_lambda))%*%t(V$vectors) 
  return(sqrt_Sigma)
}

########################
# Input: beta_true is the true coefficient of the linear model; Kdim is the dimension of a subspace that contains beta_true, 
#        p is the dimension of the predictor
#Output: the square of the covariance matrix
sqrt_Sigma3=function(p,Kdim,beta_true)
{ 
  rd=matrix(rnorm(p*p),ncol=p)  # generate random eigenvector
  Btrue = matrix(rnorm(p*Kdim),ncol=Kdim)  # generate random eigenvectors
  Btrue[,1] = beta_true

  Q.beta=diag(p)-Btrue%*%(ginv(t(Btrue)%*%Btrue))%*%t(Btrue)
  V_eig=eigen(Q.beta%*%rd%*%t(rd)%*%Q.beta)
  v_lambda=exp(-0.01*c(p:1))  
  eigvalue_lambda=p*v_lambda/(sum(v_lambda))
  V=V_eig$vectors[,c(1:p)]

  sqrt_Sigma=V%*%diag(sqrt(eigvalue_lambda))%*%t(V) 
  
  return(sqrt_Sigma)
}