from pathlib import Path
from collections import Counter
import random
import numpy as np

from src.utils import find_project_root
import pandas as pd
from src.language_mappings import FLEURS_TO_FLORES
import evaluate

sacrebleu = evaluate.load("sacrebleu")


try:
    PROJECT = find_project_root(__file__)
except NameError:  # Handle interactive use
    PROJECT = find_project_root(str(Path("./dummy_file.py").absolute()))


DATA_DIR = PROJECT / "data"
FLEURS_ASR_DIR = DATA_DIR / "flores-fleurs_asr"
LOG_DIR = PROJECT / "logs"


def get_split(splits: list[str]) -> str:
    """
    Determine a single split label ('train', 'dev', or 'test') from a list of splits.

    Parameters
    ----------
    splits : list[str]
        A list of split annotations (e.g., ['train', 'dev', 'dev', 'test']).

    Returns
    -------
    str
        The resolved split type. If there's a single consistent split,
        returns it directly. If 'train' is present, returns 'train'.
        If all are 'dev', returns 'dev', similarly for 'test'. Otherwise,
        randomly selects one of the most common splits.
    """
    if "train" in splits:
        return "train"
    elif all(x == "dev" for x in splits):
        return "dev"
    elif all(x == "test" for x in splits):
        return "test"
    else:
        counter = Counter(splits)
        split, _ = random.sample(counter.most_common(), 1)[0]
        return split


def load_data() -> dict[str, pd.DataFrame]:
    """
    Load ASR datasets from parquet files, associate them to language codes,
    and resolve their overall split annotations.

    Returns
    -------
    dict[str, pd.DataFrame]
        A dictionary with language codes as keys and corresponding DataFrames as values.
        Each DataFrame includes a 'split' column indicating 'train', 'dev', or 'test'.
    """
    lang = {}
    for lang_parquet_file in FLEURS_ASR_DIR.glob("*.parquet"):
        df = pd.read_parquet(lang_parquet_file)
        df["split"] = df.split.apply(get_split)
        lang[FLEURS_TO_FLORES[lang_parquet_file.stem]] = df
    return lang


def analyse_asr(lang: dict[str, pd.DataFrame]) -> pd.DataFrame:
    """
    Analyze ASR performance metrics (CER, WER) for both whisper and seamlessm4t models.

    Parameters
    ----------
    lang : dict[str, pd.DataFrame]
        A dictionary mapping language codes to their respective DataFrames.
        Each DataFrame is expected to have columns with ASR metrics, such as
        'whisper_asr_cer', 'whisper_asr_wer', 'seamlessm4t_asr_cer',
        'seamlessm4t_asr_wer'.

    Returns
    -------
    pd.DataFrame
        A DataFrame summarizing the mean and standard deviation of ASR metrics
        for each language and split (train, dev, test).
    """
    records = []
    for lang_, df in lang.items():
        lang_records = {}
        lang_records["lang"] = lang_
        for split in ("train", "dev", "test"):
            mask = df.split == split
            df_ = df.loc[mask]
            for model in ("whisper", "seamlessm4t"):
                for asr_metric in ("cer", "wer"):
                    asr_values = df_[f"{model}_asr_{asr_metric}"].apply(np.mean)
                    avg = float(asr_values.mean())
                    std = float(asr_values.std())
                    lang_records[f"{split}_{model}_asr_{asr_metric}_mean"] = avg
                    lang_records[f"{split}_{model}_asr_{asr_metric}_std"] = std
        records.append(lang_records)
    df = pd.DataFrame.from_records(records)
    return df


def compute_bleu(language: str, dfs: dict[str, pd.DataFrame], by_split: bool = True):
    """
    Compute BLEU scores for translations generated by ASR models (whisper and seamlessm4t).
    The target reference is English (eng_Latn), and predicted translations are from
    another language. For each language, the best translation hypothesis (based on minimal CER)
    is chosen from the arrays of translations.

    Parameters
    ----------
    language : str
        Language code (Flores style) for which BLEU is being computed.
    dfs : dict[str, pd.DataFrame]
        Dictionary mapping language codes to DataFrames containing ASR and translation data.
    by_split : bool, optional
        If True, compute BLEU not only for the entire dataset but also separately for each split.

    Returns
    -------
    dict
        Dictionary with overall BLEU scores and (if by_split=True) BLEU scores per split,
        for both whisper and seamlessm4t models.
    """
    eng_Latn = dfs["eng_Latn"]
    language_df = dfs[language]

    eng_columns = ["sentence", "fleurs_id"]

    language_df["seamlessm4t_translation"] = language_df.apply(
        lambda row: row["seamlessm4t_asr_translation"][
            np.argmin("seamlessm4t_asr_cer")
        ],
        axis=1,
    )
    language_df["whisper_translation"] = language_df.apply(
        lambda row: row["whisper_asr_translation"][np.argmin("whisper_asr_cer")],
        axis=1,
    )
    lang_columns = [
        "fleurs_id",
        "seamlessm4t_translation",
        "whisper_translation",
        "sentence",
        "split",
    ]
    merged = pd.merge(
        eng_Latn.loc[:, eng_columns],
        language_df.loc[:, lang_columns],
        on="fleurs_id",
        how="inner",
        suffixes=("_eng_Latn", f"_{language}"),
    )
    references = [[line] for line in merged["sentence_eng_Latn"].tolist()]
    seamlessm4t_predictions = merged["seamlessm4t_translation"].tolist()
    seamlessm4t_result = sacrebleu.compute(
        predictions=seamlessm4t_predictions, references=references
    )["score"]
    whisper_predictions = merged["whisper_translation"].tolist()
    whisper_result = sacrebleu.compute(
        predictions=whisper_predictions, references=references
    )["score"]

    out = {
        "lang": language,
        "seamlessm4t": seamlessm4t_result,
        "whisper": whisper_result,
    }
    if by_split:
        for split in merged.split.unique():
            mask = merged.split == split
            df_ = merged.loc[mask]
            references = [[line] for line in df_["sentence_eng_Latn"].tolist()]
            seamlessm4t_predictions = df_["seamlessm4t_translation"].tolist()
            seamlessm4t_result = sacrebleu.compute(
                predictions=seamlessm4t_predictions, references=references
            )["score"]
            whisper_predictions = df_["whisper_translation"].tolist()
            whisper_result = sacrebleu.compute(
                predictions=whisper_predictions, references=references
            )["score"]
            out[f"seamlessm4t_{split}"] = seamlessm4t_result
            out[f"whisper_{split}"] = whisper_result
    return out


def analyse_bleu(langs: dict[str, pd.DataFrame]):
    """
    Compute BLEU scores for all languages (except English) and summarize in a DataFrame.

    Parameters
    ----------
    langs : dict[str, pd.DataFrame]
        Dictionary mapping language codes to their respective DataFrames.

    Returns
    -------
    pd.DataFrame
        DataFrame containing BLEU scores (overall and by split) for each language
        and each ASR model.
    """
    records = []
    for language in langs.keys():
        if language != "eng_Latn":
            records.append(compute_bleu(language, langs))
    translation_eval = pd.DataFrame.from_records(records)
    return translation_eval


def main():
    """
    Main entry point:
    1. Load data and resolve splits.
    2. Analyze ASR metrics and save results.
    3. Analyze BLEU scores for translations and save results.
    """
    lang = load_data()
    asr_df = analyse_asr(lang)
    asr_df.to_csv(str(LOG_DIR / "asr_analysis.csv"), index=False)
    bleu_df = analyse_bleu(lang)
    bleu_df.to_csv(str(LOG_DIR / "bleu_analysis.csv"), index=False)
