import os
from .env import *
import torch
from PIL.Image import Image
from .imgpatch import ImagePatch
from . import imgpatch_local
from io import BytesIO
import base64

from openai import AsyncOpenAI
import dotenv
dotenv.load_dotenv()

client = None
def initialize_models():
    global client
    client = AsyncOpenAI(
        api_key=os.getenv("OPENAI_API_KEY")
    )
    imgpatch_local.initialize_object()
    imgpatch_local.initialize_clip()

THRESH_OBJ_DET = 0.125

def image_to_base64(image: Image) -> str:
    buffered = BytesIO()
    image.save(buffered, format="PNG")
    return base64.b64encode(buffered.getvalue()).decode("utf-8")

async def find(self : ImagePatch, object_name: str) -> IList[ImagePatch]:
    predictions = imgpatch_local.object_detector(
        self.cropped_image,
        candidate_labels=[object_name],
    )
    bbs = [
        (pred['box']['xmin'], pred['box']['ymin'], pred['box']['xmax'], pred['box']['ymax'])
        for pred in predictions if pred['score'] > THRESH_OBJ_DET
    ]
    return tuple(
        self.crop(*bb)
        for bb in bbs
    )
async def verify_property(image_patch, object_name: str, attribute: str) -> bool:
    image_b64 = image_to_base64(image_patch.cropped_image)
    prompt = f"""
You are given an image of a {object_name}. Determine whether the attribute "{attribute}" applies to the object in the image. 
Answer with "Yes" or "No".
"""
    # Call GPT-4o with the image and the prompt
    response = await client.chat.completions.create(
        model="gpt-4o-mini",
        messages=[
            {"role": "system", "content": "You are a helpful vision-language assistant."},
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": prompt},
                    {"type": "image_url", "image_url": {"url": f"data:image/png;base64,{image_b64}"}}
                ]
            }
        ],
        max_tokens=10,
        temperature=0
    )

    answer = response.choices[0].message.content.strip().lower()
    return "yes" in answer

async def best_text_match(image_patch, option_list: list[str], prefix: Optional[str] = None) -> str:
    image_b64 = image_to_base64(image_patch.cropped_image)

    # Build prompt options
    if prefix is not None:
        option_list_to_use = [f"{prefix} {opt}" for opt in option_list]
    else:
        option_list_to_use = option_list

    options_text = "\n".join([f"{i+1}. {opt}" for i, opt in enumerate(option_list_to_use)])
    
    prompt = f"""You are given an image and a list of possible descriptions. 
Choose the option that best matches the content of the image.
Return only the number of the best option.

Options:
{options_text}
"""

    response = await client.chat.completions.create(
        model="gpt-4o-mini",
        messages=[
            {"role": "system", "content": "You are a helpful assistant for image classification."},
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": prompt},
                    {"type": "image_url", "image_url": {"url": f"data:image/png;base64,{image_b64}"}}
                ]
            }
        ],
        max_tokens=5,
        temperature=0
    )

    reply = response.choices[0].message.content.strip()

    # Extract selected index
    try:
        selected_index = int(reply.split()[0]) - 1
        return option_list[selected_index]
    except (ValueError, IndexError):
        raise ValueError(f"Could not parse GPT-4o response: '{reply}'")

async def simple_query(self : ImagePatch, query: str) -> str:
    image_b64 = image_to_base64(self.cropped_image)
    stream = await client.chat.completions.create(
        model="gpt-4o-mini",
        messages=[
            {"role": "user", "content": [
                {"type": "image_url", "image_url": {"url": f"data:image/png;base64,{image_b64}"}},
                {"type": "text", "text": f"{query} Respond with a short answer only — no explanation."}
            ]}
        ],
        temperature=0,
    )
    answer = stream.choices[0].message.content
    if answer is None:
        answer = ""
    return answer

exists = imgpatch_local.exists
