import os
import sys
import dotenv
from tqdm import tqdm
import random
import datasets
from openai import OpenAI
import traceback
import argparse

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))
from utils import (
    write_txt,
    read_txt,
    load_prompt,
    response_to_py_program,
    response_to_epics_program,
    get_prog_filepaths_from_dirs
)

dotenv.load_dotenv()
client = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))
KINDS_DIR = os.path.join(os.path.dirname(__file__), f"../datasets/gqa")

def load_gqa(split):
    dataset_images = datasets.load_dataset("lmms-lab/GQA", f"{split}_all_images", split=split)
    dataset_instructions = datasets.load_dataset("lmms-lab/GQA", f"{split}_all_instructions", split=split)
    image_mappings = {item["id"]: item["image"] for item in dataset_images}
    random.seed(2025)
    num_sample=1000
    indices = random.sample(range(len(dataset_instructions)), len(dataset_instructions))[0:num_sample]
    return image_mappings, dataset_instructions, indices


def translate_py(python_dir, prompt_path, save_dir, model_name):
    prompt = load_prompt(prompt_path)
    python_prog_filepaths = get_prog_filepaths_from_dirs([python_dir])
    
    print(f"📂 Saving results to: {save_dir}")
    print(f"📜 Loaded prompt from: {prompt_path}")
    os.makedirs(save_dir, exist_ok=True)
    
    i = 0
    for py_filepath in tqdm(python_prog_filepaths):
        i += 1
        if i < 90:
            continue
        filename = os.path.basename(py_filepath)
        program_path = os.path.join(save_dir, filename)
        err_path = os.path.join(save_dir, f"{filename.split('.')[0]}.err")
        
        py_prog = read_txt(py_filepath)
        translate_prompt = prompt.replace("{new_python_code}", py_prog)

        try:
            response = client.chat.completions.create(
                model=model_name,
                messages=[
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "text",
                                "text": translate_prompt,
                            },
                            # {
                            #     "type": "image_url",
                            #     "image_url": {"url": f"data:image/jpeg;base64,{base64_image}"},
                            # }
                        ]
                    }
                ],
                temperature=0,
            )
        except Exception as e:
            write_txt(err_path, "\n".join(traceback.format_exception(e)))
            continue
        
        try:
            model_output = response.choices[0].message.content.strip()
            program = response_to_epics_program(model_output)
            write_txt(program_path, program)
        except Exception as e:
            write_txt(err_path, "\n".join(traceback.format_exception(e)))
            traceback.print_exception(e)

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-p', '--python_dir', 
        help='Python programs used for translation', 
        required = True
    )
    parser.add_argument('-e', '--epic_dir', help='Directory to save EPICs', required=True)
    parser.add_argument('-prompt', '--prompt_path', help='Path to the prompt file', required=True)
    args = parser.parse_args()

    translate_py(python_dir=args.python_dir, prompt_path=args.prompt_path, save_dir=args.epic_dir, model_name="gpt-4o-mini")