#!/usr/bin/env python

import os
import sys
import traceback
import pathlib
import time
import argparse
import ast

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), "..")))
from epic import *
from run_utils import *
from utils import (
    create_dir_w_timestamp,
    get_txt_filepaths_from_dirs,
    get_prog_filepaths_from_dirs,
    read_txt,
    write_jsonl,
    write_epics,
    write_txt,
    write_json,
    load_json
)
from epic.epics_syntax import to_str as epic_to_str

# DATASETS_DIRNAME = f"{pathlib.Path(__file__).parent.resolve()}/../datasets"
KINDS_DIR = os.path.join(os.path.dirname(__file__), f"../datasets/gqa")

parser = argparse.ArgumentParser()
parser.add_argument(
    '-k', '--kind',
    help='Target kind',
    choices=list(f for f in os.listdir(KINDS_DIR) if os.path.isdir(os.path.join(KINDS_DIR, f))), required = True
)
args = parser.parse_args()

# Directory for vipergpt programs
KIND = args.kind
DIRNAME = f"{KINDS_DIR}/{KIND}"

# Load vipergpt programs
filepaths = get_prog_filepaths_from_dirs([f"{DIRNAME}/progs_py"])

# TRANS_DIRNAME = f"{DIRNAME}/translations"
TRANS_DIRNAME = f"{DIRNAME}/progs_epic"
META_DIRNAME = f"{DIRNAME}/info_compilation"
os.makedirs(TRANS_DIRNAME, exist_ok=True)

CONTEXT = make_context(imgpatch_test, False, False)

results = []
for i, fn_path in enumerate(filepaths, 1):
    filename = os.path.basename(fn_path)
    print(f"[{i}/{len(filepaths)}] Processing: {filename}")

    filename_no_ext = os.path.splitext(filename)[0]

    trans_filepath = os.path.join(TRANS_DIRNAME, f"{filename_no_ext}.prog")
    err_filepath = os.path.join(TRANS_DIRNAME, f"{filename_no_ext}.err")
    meta_filepath = os.path.join(META_DIRNAME, f"{filename_no_ext}.json")

    py_code = read_txt(fn_path)

    try:
        execute_command = get_py_exec_command(py_code, filename, CONTEXT)
        if execute_command is None:
            print(f"    Skipping {filename} (no execute_command found)")
            continue

        print(f"    Translating {filename} to EPIC ...")
        symbol_next_id = syntax._symbol_next_id
        start = time.perf_counter_ns()
        #func_ast = ast.parse(py_code, filename=filename, mode='exec', type_comments=True)
        #epic_expr = translator.translateToEpic(func_ast, filename, 1)
        #epics_expr = epics_syntax.from_epic(epic_expr)
        #epic_final = translator.translationFinalize(epic_expr, [None], epics_vipergpt.make_mappings(CONTEXT.METHODS))
        epics_expr = epics_syntax.from_python_str(py_code, fn_path)
        epic_final = epics_vipergpt.finalize(epics_expr, [None], epics_vipergpt.make_mappings(CONTEXT.METHODS))
        elapsed = time.perf_counter_ns() - start
        # write_epics(trans_filepath, epics_expr)
        write_txt(trans_filepath, epic_to_str(epics_expr))
        
        with open(meta_filepath, "w") as f:
            json.dump({
                "time_ns" : elapsed,
            }, f)

        try:
            os.remove(err_filepath)
        except FileNotFoundError:
            pass
    except Exception as e:
        print(f"        Failed to translate {filename} to EPIC: {e}")
        write_txt(err_filepath, "\n".join(traceback.format_exception(e)))
        traceback.print_exc()
        try:
            os.remove(meta_filepath)
        except FileNotFoundError:
            pass
        try:
            os.remove(trans_filepath)
        except FileNotFoundError:
            pass
    finally:
        syntax._symbol_next_id = symbol_next_id
