import ast

class ControlFlowAnalyzer(ast.NodeVisitor):
    def __init__(self):
        self.has_loop = False
        self.has_break_continue = False
        self.nesting_depth = 0
        self.max_nesting_depth = 0
    
    def visit_For(self, node):
        self.has_loop = True
        self._increase_nesting(node)

    def visit_While(self, node):
        self.has_loop = True
        self._increase_nesting(node)

    def visit_Break(self, node):
        self.has_break_continue = True

    def visit_Continue(self, node):
        self.has_break_continue = True

    def _increase_nesting(self, node):
        self.nesting_depth += 1
        self.max_nesting_depth = max(self.max_nesting_depth, self.nesting_depth)
        self.generic_visit(node)
        self.nesting_depth -= 1

def has_complex_control_flow(program: str) -> bool:
    """
    Returns True if the program has loops, break/continue, or nested control structures.
    """ 
    try:
        tree = ast.parse(program)
    except SyntaxError as e:
        raise SyntaxError(f"Invalid Python code: {e}")
    
    analyzer = ControlFlowAnalyzer()
    analyzer.visit(tree)
    return (
        analyzer.has_loop or 
        analyzer.has_break_continue or 
        analyzer.max_nesting_depth > 1
    )

def has_loop(program: str) -> bool:
    """
    Returns True if the program contains at least one for or while loop.
    """
    try:
        tree = ast.parse(program)
    except SyntaxError as e:
        raise SyntaxError(f"Invalid Python code: {e}")
    
    class LoopChecker(ast.NodeVisitor):
        def __init__(self):
            self.found = False
        
        def visit_For(self, node):
            self.found = True
        
        def visit_While(self, node):
            self.found = True
    
    checker = LoopChecker()
    checker.visit(tree)
    return checker.found

def has_single_return(program: str) -> bool:
    """
    Returns True if the program contains exactly one return statement.
    """
    try:
        tree = ast.parse(program)
    except SyntaxError as e:
        raise SyntaxError(f"Invalid Python code: {e}")

    return_count = sum(isinstance(node, ast.Return) for node in ast.walk(tree))
    return return_count == 1