/**
 * @file GPLearning.cpp
 *
 * @brief
 * GP Learning: create information from measures for GP Prediction
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 *
 * @date
 * Modified on 2023/02/10
 *

 */

////////////////////////////////////////////////////////////////////////////////

#include "GPLearning.h"
#include "GPPrediction.h"
#include "GPTypes.cpp"


//--------------------------------------------------------------------------
// this first version of kalmanStep is for reduced-rank GP without input noise
// (posValueVect does not contain the variance associated to the point studied)
void GPLearning::kalmanStep(const posValueVect & listCenteredPosInCubeAndValue,
                                    const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                                    const double sigmaNoiseSquareModel,
                                    const VectorDimInputd & LxLyLz,
                                    Eigen::VectorXd & learningPrediction,
                                    Eigen::MatrixXd & learningCovar)
{
 std::vector<VectorDimInputd> listCenteredPosInCube;
 std::vector<VectorDimOutputd> listValue;
 separateData(listCenteredPosInCubeAndValue, listCenteredPosInCube, listValue);
 const unsigned int nbDatapoints     = listCenteredPosInCubeAndValue.size();
 const unsigned int nbDatapoints_X_dimOutput = nbDatapoints*dimOutput ;
 // stack all data in one vector
 Eigen::VectorXd vecValueMeasures = Eigen::VectorXd(nbDatapoints*dimOutput) ;
 for (unsigned int i = 0 ; i < nbDatapoints ; i++)
 {
   vecValueMeasures.segment<dimOutput>(dimOutput*i) = listValue.at(i);
 }
 // depending on the number of data, one formula is more efficient than the other.
 // Both yield exactly the same result up to numerical errors
 Eigen::MatrixXd K;
 Eigen::MatrixXd KT;
 Eigen::MatrixXd S;
 Eigen::MatrixXd invS;
 const Eigen::MatrixXd Phi  = matrixPhi(listCenteredPosInCube, basisOfEigenfunctions, LxLyLz);
 const Eigen::MatrixXd PhiT = Phi.transpose();
 const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
 if(dimInput+nbEigenfunctions < nbDatapoints_X_dimOutput)
 {
   const Eigen::MatrixXd I_3plusm = Eigen::MatrixXd::Identity(dimInput+nbEigenfunctions, dimInput+nbEigenfunctions);
   Eigen::MatrixXd invLearningCovar = learningCovar.llt().solve(I_3plusm);
   S = PhiT*Phi + sigmaNoiseSquareModel*invLearningCovar;
   invS = S.llt().solve(I_3plusm);
   K = invS*PhiT;

   learningPrediction = learningPrediction + K*(vecValueMeasures - Phi*learningPrediction);
   learningCovar = sigmaNoiseSquareModel*invS;
 }
 else
 {
   const Eigen::MatrixXd I_dimOutputn = Eigen::MatrixXd::Identity(nbDatapoints_X_dimOutput, nbDatapoints_X_dimOutput);
   S = Phi*learningCovar*PhiT + sigmaNoiseSquareModel*I_dimOutputn;
   invS = S.llt().solve(I_dimOutputn);
   K = learningCovar*PhiT*invS;
   Eigen::MatrixXd KT = K.transpose();

   learningPrediction = learningPrediction + K*(vecValueMeasures - Phi*learningPrediction);
   learningCovar = learningCovar - K*S*KT;

 }
 learningCovar = (learningCovar + learningCovar.transpose())/2;
}
//--------------------------------------------------------------------------
// this second version of kalmanStep is for reduced-rank GP with input noise,
// this is the implementation of RRONIG
bool GPLearning::kalmanStep(const posCovValueVect & listCenteredPosInCubeAndValue,
                                    const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                                    const double sigmaNoiseSquareModel,
                                    const double sigmaLinSquare,
                                    const double sigmaSESquare,
                                    const double lSESquare,
                                    const VectorDimInputd & LxLyLz,
                                    Eigen::VectorXd & learningPrediction,
                                    Eigen::MatrixXd & learningCovar)
{
 std::vector<VectorDimInputd> listCenteredPosInCube;
 std::vector<MatrixDimInputd> cov_pos_v1;
 std::vector<VectorDimOutputd> listValue;
 separateData(listCenteredPosInCubeAndValue, listCenteredPosInCube, listValue, cov_pos_v1);
 const unsigned int nbDatapoints     = listCenteredPosInCubeAndValue.size();
 const unsigned int nbDatapoints_X_dimOutput = nbDatapoints*dimOutput ;

 // check the data and stack all data in one vector
 bool problem = false ;
 const double maxAmpliValue = 100;
 Eigen::VectorXd vecValueMeasures = Eigen::VectorXd(nbDatapoints*dimOutput) ;
std::vector<MatrixDimInputd> cov_pos;
std::vector<Eigen::MatrixXd> esp_grad_B;
VectorDimInputd epsilon = 1e-6*VectorDimInputd::Ones();
 for (unsigned int i = 0 ; i < nbDatapoints ; i++)
 {
   VectorDimOutputd valueValue = listValue.at(i);
     vecValueMeasures.segment<dimOutput>(dimOutput*i) = listValue.at(i);
     const MatrixDimInputd covPosExtracted = cov_pos_v1.at(i);
     cov_pos.push_back(covPosExtracted);
     const Eigen::MatrixXd espGradientB = matrixGradientBFromLearningPred(listCenteredPosInCube.at(i)+epsilon,basisOfEigenfunctions, LxLyLz, learningPrediction);
     esp_grad_B.push_back(espGradientB);
 }

 const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
 const Eigen::MatrixXd Phi  = matrixPhi(listCenteredPosInCube, basisOfEigenfunctions, LxLyLz);
 const Eigen::MatrixXd PhiT = Phi.transpose();
 Eigen::MatrixXd K;
 Eigen::MatrixXd KT;
 Eigen::MatrixXd S;
 Eigen::MatrixXd invS;
 const Eigen::MatrixXd I_dimOutputn = Eigen::MatrixXd::Identity(nbDatapoints_X_dimOutput, nbDatapoints_X_dimOutput);

 const Eigen::MatrixXd DVar = matrixDVarFromLearningPredAndCovar(listCenteredPosInCube, cov_pos,
                                                                basisOfEigenfunctions, LxLyLz, learningPrediction, learningCovar);

     Eigen::MatrixXd Theta = computeThetaFromZeta(listCenteredPosInCube, cov_pos, basisOfEigenfunctions, LxLyLz);

     Eigen::MatrixXd ThetaT = Theta.transpose();
     Eigen::MatrixXd uncertaintyScalarProduct(dimOutput, dimOutput);
  if (dimOutput == 1)
  {
    // for 1D, the matrix and its transpose are equal
    uncertaintyScalarProduct = (Phi*learningCovar*ThetaT);
  }
  else
  {
      uncertaintyScalarProduct = 0.5*(Phi*learningCovar*ThetaT+ Theta*learningCovar.transpose()*ThetaT);
  }
     Eigen::MatrixXd uncertaintyLoc = DVar + uncertaintyScalarProduct;

    bool issueVar = false;
    for (unsigned int i = 1; i<nbDatapoints_X_dimOutput ;i++)
    {
      std::complex<double> eigenvalueStudied = (uncertaintyLoc).eigenvalues()(i);
      if (std::real<double>(eigenvalueStudied) < 0 || std::abs(std::imag<double>(eigenvalueStudied)) > 1e-10)
      {
        std::cout << "Issue with variance: negative eigenvalue" << std::endl;
        issueVar = true;
      }
    }
      S = Phi*learningCovar*PhiT + sigmaNoiseSquareModel*I_dimOutputn + uncertaintyLoc ;

       invS = S.llt().solve(I_dimOutputn);
      K = learningCovar*(PhiT + 0.5*ThetaT)*invS;
    learningPrediction = learningPrediction + K*(vecValueMeasures - (Phi+0.5*Theta)*learningPrediction);

     KT = K.transpose();
     learningCovar = learningCovar - K*S*KT;

   learningCovar = (learningCovar + learningCovar.transpose())/2;

   // test for coherence with symmetric-definite variance
   double minEigenvalCovar = learningCovar.eigenvalues()(0).real();
   for (unsigned int i = 0; i<size(learningCovar.eigenvalues()); i++)
   {
     minEigenvalCovar = std::min(minEigenvalCovar, learningCovar.eigenvalues()(i).real());
     if (learningCovar.eigenvalues()(i).imag())
     {
       minEigenvalCovar = -1;
     }
   }
   if (minEigenvalCovar<0)
   {
     std::cout << "Problem: negative covariance, issue on the modeling" << std::endl;
   }
 return (!problem);
}
//--------------------------------------------------------------------------
  std::vector<VectorDimInputI> GPLearning::computeBasisOfEigenfunctions(const VectorDimInputI & nbFrequsForBasisOfEigenfunctions)
  {
    std::vector<VectorDimInputI> allEigenfunctions;

  VectorDimInputI frequs;
  unsigned int nbEigen = 1;
  for (unsigned int i = 0 ; i < dimInput ; i++)
  {
    frequs(i) = nbFrequsForBasisOfEigenfunctions(i);
    nbEigen*= nbFrequsForBasisOfEigenfunctions(i);
  }

  allEigenfunctions.resize(nbEigen);

  for (unsigned int i = 0 ; i < nbEigen ; i++)
    {
      VectorDimInputI newEigenFun;
      unsigned int indexTmp = 0;
      unsigned int coordOfIthValue = i;
      while (indexTmp < dimInput)
      {
        newEigenFun(indexTmp) = 1+(coordOfIthValue% nbFrequsForBasisOfEigenfunctions(indexTmp));
        coordOfIthValue = coordOfIthValue/ nbFrequsForBasisOfEigenfunctions(indexTmp);
        indexTmp += 1;
      }
    allEigenfunctions.at(i) = newEigenFun;
  };


   return allEigenfunctions;
  }
//--------------------------------------------------------------------------
  Eigen::MatrixXd GPLearning::matrixPhi(const std::vector<VectorDimInputd> & listCenteredPosInCube, const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz)
  {
    const unsigned int nbDatapoints    = listCenteredPosInCube.size();
    const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();

    Eigen::MatrixXd matrixPhi(dimOutput*nbDatapoints, dimInput+nbEigenfunctions);

    for (int i = 0; i < nbDatapoints; i++)
    { matrixPhi.block(dimOutput*i,0, dimOutput, dimInput+nbEigenfunctions) = PhiBlockLine(listCenteredPosInCube.at(i), basisOfEigenfunctions, LxLyLz); }

    return matrixPhi;
  }
//--------------------------------------------------------------------------
  Eigen::MatrixXd GPLearning::PhiBlockLine(const VectorDimInputd & centeredPosInCube, const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz)
  {
    const int nbEigenfunctions = basisOfEigenfunctions.size();

    Eigen::MatrixXd allEigNegativeLaplace(dimOutput,nbEigenfunctions);

    for (int i = 0; i < nbEigenfunctions; i++)
      { allEigNegativeLaplace.col(i) = eigNegLaplace(centeredPosInCube, basisOfEigenfunctions.at(i), LxLyLz); }

    Eigen::MatrixXd Phi(dimOutput, dimInput+nbEigenfunctions);
     for (unsigned int i = 0; i<dimInput; i++)
     {
            Phi.block(0,i,dimOutput,1)               = centeredPosInCube(i)*VectorDimOutputd::Ones() ; //x
     }
    ////////////////////////////////////////////////////////
    Phi.topRightCorner(dimOutput,nbEigenfunctions) = allEigNegativeLaplace ;
    return Phi;
  }
//--------------------------------------------------------------------------
VectorDimInputd GPLearning::eigNegLaplace(const VectorDimInputd & centeredPosInCube, const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz)
{
  VectorDimInputd dphi = VectorDimInputd::Zero();
  for (unsigned int i = 0; i<dimInput ; i++)
  {
    dphi(i) = partialDerivEigNegLaplace(centeredPosInCube, eigenfunction, LxLyLz, VectorDimInputI::Zero());

  }
  return dphi;
}
//--------------------------------------------------------------------------
double GPLearning::partialDerivEigNegLaplace(const VectorDimInputd & centeredPosInCube,
                                                      const VectorDimInputI & eigenfunction,
                                                      const VectorDimInputd & LxLyLz,
                                                      const VectorDimInputI & a)
{
  double partialDeriv = 1;
  for(unsigned int d = 0; d < dimInput; d++)
  {
    double omega = CST_PI*eigenfunction(d)/(2*LxLyLz(d));
    partialDeriv *= (pow(omega, a(d))*
                    pow(-1, a(d)/2)*
                    ((a(d)%2) == 0 ? std::sin(omega*(centeredPosInCube(d)+LxLyLz(d))) : std::cos(omega*(centeredPosInCube(d)+LxLyLz(d))))/
                    std::sqrt(LxLyLz(d)));
  }
  return partialDeriv;
}
//-------------------------------------------------------------------------
Eigen::VectorXd GPLearning::partialDerivEigNegLaplaceBlockLine(const VectorDimInputd & centeredPosInCube,
                                                      const std::vector<VectorDimInputI> & basisOfEigenfunctions,
                                                      const VectorDimInputd & LxLyLz,
                                                      const VectorDimInputI & a)
{
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::VectorXd partialDerivEigFunct(nbEigenfunctions);
  for (unsigned int i=0; i < nbEigenfunctions; i++)
  {
    partialDerivEigFunct(i) = partialDerivEigNegLaplace(centeredPosInCube, basisOfEigenfunctions.at(i), LxLyLz, a);
  }
  return partialDerivEigFunct;
}
//---------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::computeDgradphi_dxi(const VectorDimInputd & pos,
                          const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                          const unsigned int i)
{
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::MatrixXd dgradphi_dxiT = Eigen::MatrixXd::Zero(dimInput+nbEigenfunctions,dimOutput);
    for (unsigned int i = 0; i<dimInput; i++)
    {
      VectorDimInputI deriv = VectorDimInputI::Zero();
      deriv(i) = 1;
      dgradphi_dxiT.block(dimInput,i, nbEigenfunctions,1) = partialDerivEigNegLaplaceBlockLine(pos, basisOfEigenfunctions, LxLyLz, deriv);
    }
    const Eigen::MatrixXd dgradphi_dxi = dgradphi_dxiT.transpose();
    return dgradphi_dxi;
}
//---------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::computeVarGradientBTimesPosError(const VectorDimInputd & pos, const MatrixDimInputd & posCov,
                          const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                          const Eigen::MatrixXd & learningCovar)
{
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  MatrixDimOutputd varGradientB = MatrixDimOutputd::Zero();
  // CAUTION! here, only one pose is given, xi = xj
  std::vector<Eigen::MatrixXd> all_dgradphi_dxi;
  for (unsigned int i = 0; i<dimInput; i++)
  {
    all_dgradphi_dxi.push_back(computeDgradphi_dxi(pos, basisOfEigenfunctions, LxLyLz, i));
  }

  for (unsigned int i = 0; i<dimInput; i++)
  {
    Eigen::MatrixXd dgradphi_dxi = all_dgradphi_dxi.at(i);
    for(unsigned int j = 0; j<dimInput; j++)
    {
      Eigen::MatrixXd dgradphi_dxj = all_dgradphi_dxi.at(j);
	     varGradientB += posCov(i,j) * dgradphi_dxi*(learningCovar)*dgradphi_dxj.transpose();
    }
  }
  return varGradientB;
}
//---------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::matrixGradientBFromLearningPred(const VectorDimInputd & pos,
                          const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                          const Eigen::VectorXd & learningPredictionPos)
{
  // new version
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::MatrixXd gradientB = Eigen::MatrixXd::Zero(dimOutput,dimInput);
  // CAUTION! here, only one pose is given, xi = xj
  std::vector<Eigen::MatrixXd> all_dgradphi_dxi;
  for (unsigned int i = 0; i<dimInput; i++)
  {
      all_dgradphi_dxi.push_back(computeDgradphi_dxi(pos, basisOfEigenfunctions, LxLyLz, i));
  }

  for (unsigned int i = 0; i<dimInput; i++)
  {
    Eigen::MatrixXd dgradphi_dxi = all_dgradphi_dxi.at(i);
    gradientB.block(0,i,dimOutput,1) = dgradphi_dxi*(learningPredictionPos);
  }
  return gradientB;
}
//-------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::matrixDVarFromLearningPred(const std::vector<VectorDimInputd> & posList, const std::vector<MatrixDimInputd> & varPosList,
                          const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                        const Eigen::VectorXd & learningPrediction)
{
const unsigned int nbDatapoints = posList.size();
Eigen::MatrixXd DVar = Eigen::MatrixXd::Zero(dimOutput*nbDatapoints, dimOutput*nbDatapoints);
for (unsigned int i = 0; i<nbDatapoints; i++)
{
  const Eigen::MatrixXd gradientB = matrixGradientBFromLearningPred(posList.at(i), basisOfEigenfunctions, LxLyLz, learningPrediction);
  const MatrixDimInputd varPos = varPosList.at(i);
  // the gradient is symmetric here
 // here, Esp(grad B(x)) is supposed to be a constant. This is not true but not a bad approximation
  DVar.block(dimOutput*i, dimOutput*i, dimOutput, dimOutput) = gradientB*varPos*gradientB.transpose();
}
  return DVar;
}
//-------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::matrixDVarFromLearningPredAndCovar(const std::vector<VectorDimInputd> & posList, const std::vector<MatrixDimInputd> & varPosList,
                          const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                         const Eigen::VectorXd & learningPred, const Eigen::MatrixXd & learningCovar)
{
const unsigned int nbDatapoints = posList.size();
Eigen::MatrixXd DVar = Eigen::MatrixXd::Zero(dimOutput*nbDatapoints, dimOutput*nbDatapoints);
for (unsigned int i = 0; i<nbDatapoints; i++)
{
  const Eigen::MatrixXd varGradientBTimesPosError = computeVarGradientBTimesPosError(posList.at(i), varPosList.at(i), basisOfEigenfunctions, LxLyLz, learningCovar);
  const Eigen::MatrixXd espGradientB = matrixGradientBFromLearningPred(posList.at(i),basisOfEigenfunctions, LxLyLz, learningPred);
  const MatrixDimInputd varPos = varPosList.at(i);
  // gradient is symmetric
  DVar.block(dimOutput*i, dimOutput*i, dimOutput, dimOutput) = varGradientBTimesPosError+espGradientB*varPos*espGradientB.transpose();
}
  return DVar;
}
// //--------------------------------------------------------------------------
Eigen::VectorXd GPLearning::computeZetaBlockLine(const VectorDimInputd & pos, const MatrixDimInputd & varPos,
                          const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                          const unsigned int i)
{
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::VectorXd zetaBlockLine = Eigen::VectorXd::Zero(dimInput+nbEigenfunctions);

  for (unsigned int j = 0; j<dimInput; j++)
  {
    for (unsigned int k = 0; k<dimInput; k++)
    {
     VectorDimInputI a = VectorDimInputI::Zero();
     a(j) += 1;
     a(k) += 1;
   Eigen::MatrixXd scalarCovarProduct = varPos(j,k)*partialDerivEigNegLaplaceBlockLine(pos, basisOfEigenfunctions, LxLyLz, a);

     zetaBlockLine.segment(dimInput,nbEigenfunctions) += scalarCovarProduct;
    }
  }
  return zetaBlockLine;
}
//--------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::matrixZeta(const VectorDimInputd & pos, const MatrixDimInputd & varPos,
                          const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz)
{
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::MatrixXd zeta = Eigen::MatrixXd::Zero(dimOutput, dimInput+nbEigenfunctions);
  for (unsigned int i = 0; i<dimInput; i++)
  {
     zeta.row(i) = computeZetaBlockLine(pos, varPos, basisOfEigenfunctions, LxLyLz, i);
  }
  return zeta;
}
//--------------------------------------------------------------------------
Eigen::MatrixXd GPLearning::computeThetaFromZeta(const std::vector<VectorDimInputd> & posList,
                          const std::vector<MatrixDimInputd> & varPosList,
                          const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz)
{
  const unsigned int nbDatapoints = posList.size();
  const unsigned int nbEigenfunctions = basisOfEigenfunctions.size();
  Eigen::MatrixXd Theta(dimOutput*nbDatapoints, dimInput+nbEigenfunctions);
  for (unsigned int i = 0; i<nbDatapoints; i++)
  {
    Theta.block(dimOutput*i, 0, dimOutput, dimInput+nbEigenfunctions) = matrixZeta(posList.at(i), varPosList.at(i), basisOfEigenfunctions, LxLyLz);
  }
  return Theta;
}
//--------------------------------------------------------------------------
  double GPLearning::eigenfunctionNegativeLaplace(const VectorDimInputd & centeredPosInCube, const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz)
  {
    double eigenfunctionNegativeLaplace = 1.0;
    const double pi = CST_PI ;

     for (int d = 0; d < dimInput; d++)
       { eigenfunctionNegativeLaplace *= std::sin(pi*eigenfunction(d) * (centeredPosInCube(d) + LxLyLz(d)) / (2*LxLyLz(d))) /std::sqrt(LxLyLz(d)); }

    return eigenfunctionNegativeLaplace;
  }
//--------------------------------------------------------------------------
// spectral density
  Eigen::VectorXd GPLearning::vectorSpectralDensity(const double sigmaLinSquare, const double sigmaSESquare, const double lSESquare,
							    const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz)
  {
    const int nbEigenfunctions = basisOfEigenfunctions.size();

    Eigen::VectorXd allSpectralDensities = Eigen::VectorXd(nbEigenfunctions);
    for (int i = 0; i < nbEigenfunctions; i++)
    {
      double lambda_i = eigenvalueNegativeLaplace(basisOfEigenfunctions.at(i), LxLyLz);
      allSpectralDensities(i) = spectralDensityFunction(lambda_i, sigmaSESquare, lSESquare);
    }
    Eigen::VectorXd diagLambda = Eigen::VectorXd::Zero(dimInput+nbEigenfunctions);
     diagLambda.head<dimInput>()              = sigmaLinSquare*VectorDimInputd::Ones() ;
    diagLambda.tail(nbEigenfunctions) = allSpectralDensities ;

    return diagLambda;
}
//--------------------------------------------------------------------------
  double GPLearning::eigenvalueNegativeLaplace(const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz)
  {
    double lambda_j = 0;
    const double pi = CST_PI ;

    for (int d = 0; d < dimInput; d++)
      {
	double tmp_d = pi*eigenfunction(d)/(2*LxLyLz(d)) ;
	lambda_j += (tmp_d*tmp_d) ;
      }
     lambda_j = std::sqrt(lambda_j);
     return lambda_j;
  }
//--------------------------------------------------------------------------
   double GPLearning::spectralDensityFunction(const double omega, const double sigmaSESquare, const double lSESquare)
   {
     const double pi = CST_PI ;
 double S_SE = sigmaSESquare *std::pow(2*pi*lSESquare, (double)dimInput/2) * std::exp(-omega*omega * lSESquare/2);
     return S_SE;
   }
