/**
 * @file GPLearning.h
 *
 * @brief
 * GP Learning
 *
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 * @date
 * Modified on 2023/02/10
 */

////////////////////////////////////////////////////////////////////////////////

#ifndef GP_LEARNING_H
#define GP_LEARNING_H

#pragma once

#include <Eigen/Dense>
#include "GPTypes.h"

    class GPLearning
    {

    public:
      static void kalmanStep(const posValueVect & listCenteredPosInCubeAndValue_W,
                             const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                             const double sigmaNoiseSquareModel,
                             const VectorDimInputd & LxLyLz,
                             Eigen::VectorXd & learningPrediction,
                             Eigen::MatrixXd & learningCovar);

      static bool kalmanStep(const posCovValueVect & listCenteredPosInCubeAndValue_W,
                             const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                             const double sigmaNoiseSquareModel,
                             const double sigmaLinSquare,
                             const double sigmaSESquare,
                             const double lSESquare,
                             const VectorDimInputd & LxLyLz,
                             Eigen::VectorXd & learningPrediction,
                             Eigen::MatrixXd & learningCovar);

      // compute phi for one position and all the eigenfunctions of the model
      static Eigen::MatrixXd PhiBlockLine(const VectorDimInputd & centeredPosInCube, const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz);

      static std::vector<VectorDimInputI> computeBasisOfEigenfunctions(const VectorDimInputI & nbFrequsForBasisOfEigenfunctions);

      static Eigen::VectorXd vectorSpectralDensity(const double sigmaLinSquare, const double sigmaSESquare, const double lSESquare,
						      const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz);

    private :


      static Eigen::MatrixXd matrixPhi(const std::vector<VectorDimInputd> & listCenteredPosInCube, const std::vector<VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz);

        static VectorDimInputd eigNegLaplace(const VectorDimInputd & centeredPosInCube, const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz);


        static double partialDerivEigNegLaplace(const VectorDimInputd & centeredPosInCube,
                                                        const VectorDimInputI & eigenfunction,
                                                        const VectorDimInputd & LxLyLz,
                                                        const VectorDimInputI & a);


        static double eigenfunctionNegativeLaplace(const VectorDimInputd & centeredPosInCube, const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz);

        static double eigenvalueNegativeLaplace(const VectorDimInputI & eigenfunction, const VectorDimInputd & LxLyLz);

        static double spectralDensityFunction(const double omega, const double sigmaSESquare, const double l_SE);


        static Eigen::VectorXd partialDerivEigNegLaplaceBlockLine(const VectorDimInputd & centeredPosInCube,
                                                              const std::vector<VectorDimInputI> & basisOfEigenfunctions,
                                                              const VectorDimInputd & LxLyLz,
                                                              const VectorDimInputI & a);
        static Eigen::MatrixXd matrixDVarFromLearningPred(const std::vector<VectorDimInputd> & posList,
                    const std::vector<MatrixDimInputd> & varPosList,
                    const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                    const VectorDimInputd & LxLyLz, const Eigen::VectorXd & learningPrediction);
        static Eigen::MatrixXd matrixDVarFromLearningPredAndCovar(const std::vector<VectorDimInputd> & posList,
                    const std::vector<MatrixDimInputd> & varPosList,
                    const std::vector <VectorDimInputI> & basisOfEigenfunctions,
                    const VectorDimInputd & LxLyLz, const Eigen::VectorXd & learningPred,
                    const Eigen::MatrixXd & learningCovar);
        static Eigen::MatrixXd computeVarGradientBTimesPosError(const VectorDimInputd & pos, const MatrixDimInputd & posCov,
                    const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                    const Eigen::MatrixXd & learningCovar);
        static Eigen::MatrixXd computeDgradphi_dxi(const VectorDimInputd & pos,
                                    const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                                    const unsigned int i);
        static Eigen::MatrixXd matrixGradientBFromLearningPred(const VectorDimInputd & pos,
                                              const std::vector <VectorDimInputI> & basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                                              const Eigen::VectorXd & learningPredictionPos);
        static Eigen::VectorXd computeZetaBlockLine(const VectorDimInputd & pos, const MatrixDimInputd & varPos,
                                        const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz,
                                      const unsigned int i);
        static Eigen::MatrixXd matrixZeta(const VectorDimInputd & pos, const MatrixDimInputd & varPos,
                                              const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz);
        static Eigen::MatrixXd computeThetaFromZeta(const std::vector<VectorDimInputd> & posList,
                            const std::vector<MatrixDimInputd> & varPosList,
                            const std::vector <VectorDimInputI> &  basisOfEigenfunctions, const VectorDimInputd & LxLyLz);
  };



#endif
