/**
 * @file GPPrediction.cpp
 *
 * @brief
 * GP Prediction: store GP eigenfunctions (reduced-rank GP) and predicts the field
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 *
 *
 * @date
 * Modified on 2023/02/10
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "GPPrediction.h"
#include <Eigen/Dense>

#define CHI2_XX_THRESHOLD 12.84



//--------------------------------------------------------------------------
  GPPrediction::GPPrediction(const std::vector<VectorDimInputI> & a_basisOfEigenfunctions,
			   const VectorDimInputd & a_LxLyLz,
         const VectorDimInputd & a_LxLyLzData,
			   const VectorDimInputd & a_LxLyLzExtended,
			   const VectorDimInputd & a_pos0,
			   const double a_sigmaLinSquare, const double a_sigmaSESquare,
			   const double a_lSESquare, const double a_sigmaNoiseSquareModel,
         const std::map<triplet,CubeModel,ComparisonTriplet> & a_mapCubesModel)
    : _sigmaLinSquare(a_sigmaLinSquare)
    , _sigmaSESquare(a_sigmaSESquare)
    , _lSESquare(a_lSESquare)
    , _sigmaNoiseSquareModel(a_sigmaNoiseSquareModel)
    , _basisOfEigenfunctions( a_basisOfEigenfunctions)
    , _LxLyLz( a_LxLyLz)
    , _LxLyLzData( a_LxLyLzData)
    , _LxLyLzExtended(a_LxLyLzExtended)
    , _pos0(a_pos0)
    , _apply_pos0_offset(true)
    , _cubes(a_mapCubesModel) {};
//--------------------------------------------------------------------------
  GPPrediction::GPPrediction(const VectorDimInputI & a_nbFrequsForBasisOfEigenfunctions,
         const VectorDimInputd & a_dimCubes,
			   const VectorDimInputd & a_pos0,
			   const double a_sigmaLinSquare, const double a_sigmaSESquare,
			   const double a_lSESquare, const double a_sigmaNoiseSquareModel)
    : _sigmaLinSquare(a_sigmaLinSquare)
    , _sigmaSESquare(a_sigmaSESquare)
    , _lSESquare(a_lSESquare)
    , _sigmaNoiseSquareModel(a_sigmaNoiseSquareModel)
    , _basisOfEigenfunctions(GPLearning::computeBasisOfEigenfunctions(a_nbFrequsForBasisOfEigenfunctions))
    , _LxLyLz( a_dimCubes/2)
// the cube is larger for the learning to have a smoother border on the sides of the different cubes,
// as the space should be divided in several smaller cubes in order to cover huge spaces
    , _LxLyLzData( _LxLyLz) // *1.2)
    , _LxLyLzExtended(_LxLyLz) // *1.5)
    // , _LxLyLzData( _LxLyLz + 0.5*std::sqrt(_lSESquare)*VectorDimInputd::Ones())
    // , _LxLyLzExtended(_LxLyLz + (std::sqrt(2.0*_lSESquare) + 0.5*std::sqrt(_lSESquare))*VectorDimInputd::Ones())
    , _pos0(a_pos0)
    , _apply_pos0_offset(true)
    , _cubes(std::map<triplet,CubeModel,ComparisonTriplet>())
  {};

//
void GPPrediction::update(const posValueVect & pos_and_value)
{
std::vector<VectorDimInputd> pos;
std::vector<VectorDimOutputd> value;
separateData(pos_and_value, pos, value);
indexInCubeMap allCubes = separateCubes(pos);
for (indexInCubeMap::const_iterator it_cube=allCubes.begin(); it_cube!=allCubes.end(); ++it_cube)
{
  posValueVect listCenteredPosInCubeAndValue;
  std::vector<unsigned int> indices = it_cube->second;
  for (std::vector<unsigned int>::const_iterator it_index = indices.begin(); it_index != indices.end(); it_index++)
  {
    listCenteredPosInCubeAndValue.push_back(getCenteredPosInCube(pos_and_value.at(*it_index), it_cube->first));
  }
  updateCube(it_cube->first, listCenteredPosInCubeAndValue);
}
}
//--------------------------------------------------------------------------
void GPPrediction::update(const posCovValueVect & pos_cov_p_value)
{
std::vector<VectorDimInputd> pos;
// cov_p_R should be of dim 6
std::vector<MatrixDimInputd> cov_p_R;
std::vector<VectorDimOutputd> value;
separateData(pos_cov_p_value, pos, value, cov_p_R);
indexInCubeMap allCubes = separateCubes(pos);
for (indexInCubeMap::const_iterator it_cube=allCubes.begin(); it_cube!=allCubes.end(); ++it_cube)
{
  posCovValueVect listCenteredPosInCubePosCovAndValue;
  std::vector<unsigned int> indices = it_cube->second;
  for (std::vector<unsigned int>::const_iterator it_index = indices.begin(); it_index != indices.end(); it_index++)
  {
    listCenteredPosInCubePosCovAndValue.push_back(getCenteredPosInCube(pos_cov_p_value.at(*it_index), it_cube->first));
  }
  updateCube(it_cube->first, listCenteredPosInCubePosCovAndValue);
}
}
//--------------------------------------------------------------------------
void GPPrediction::update(const VectorDimInputd & pos, const VectorDimOutputd & value)
{
try
{
  posValueVect wrpList(1,posValueStruct(pos, value));
  update(wrpList);
}
catch(const std::runtime_error& e)
{
  std::cerr << e.what() << std::endl;
  throw std::runtime_error ("argument given to map update are incorrect");
}
}
//--------------------------------------------------------------------------
void GPPrediction::update(const VectorDimInputd & pos, const MatrixDimInputd cov_p, const VectorDimOutputd & value)
{
try
{
  posCovValueVect wrpList(1, posCovValueStruct(pos, cov_p, value));
  update(wrpList);
}
catch(const std::runtime_error& e)
{
  std::cerr << e.what() << std::endl;
  throw std::runtime_error ("argument given to map update are incorrect");
}

}
//--------------------------------------------------------------------------
void GPPrediction::updateCube(const triplet & macroCoordinates, const posValueVect & listCenteredPosInCubeAndValue)
{
  const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it = _cubes.find(macroCoordinates);
  bool cubeIsNew = !(checkModelOfCubeExists(it));
  if (cubeIsNew)
  { initCube(macroCoordinates); }
  std::cout << "init kalman step" << std::endl;
  GPLearning::kalmanStep(listCenteredPosInCubeAndValue,
                                  _basisOfEigenfunctions,
                                  _sigmaNoiseSquareModel,
                                  _LxLyLzExtended,
                                  _cubes[macroCoordinates].learningPrediction,
                                  _cubes[macroCoordinates].learningCovar);
}
//--------------------------------------------------------------------------
void GPPrediction::updateCube(const triplet & macroCoordinates, const posCovValueVect & listCenteredPosInCubePosCovAndValue)
{
  const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it = _cubes.find(macroCoordinates);
  bool cubeIsNew = !(checkModelOfCubeExists(it));
  if (cubeIsNew)
  {    initCube(macroCoordinates); }
  bool valid = GPLearning::kalmanStep( listCenteredPosInCubePosCovAndValue,
                                                _basisOfEigenfunctions,
                                                _sigmaNoiseSquareModel,
                                                _sigmaLinSquare,
                                                _sigmaSESquare,
                                                _lSESquare,
                                                _LxLyLzExtended,
                                                _cubes[macroCoordinates].learningPrediction,
                                                _cubes[macroCoordinates].learningCovar);
  if (!valid)
    { std::cerr << "== GPPrediction == updateCube, issue with data, cannot update cube !" ;}
}
//--------------------------------------------------------------------------
  void GPPrediction::initCube(const triplet & macroCoordinates)
  {
std::cout << "initialize cube " << (macroCoordinates) << std::endl;

    const unsigned int nbEigenfunctions = _basisOfEigenfunctions.size();
    Eigen::VectorXd learningPrediction   = Eigen::VectorXd::Zero(dimInput+nbEigenfunctions);
    const Eigen::VectorXd Lambda         = GPLearning::vectorSpectralDensity(_sigmaLinSquare,
                                                                                     _sigmaSESquare,
                                                                                     _lSESquare,
                                                                                     _basisOfEigenfunctions,
                                                                                     _LxLyLzExtended);
  //      std::cout << "Lambda " << Lambda << std::endl;
    Eigen::MatrixXd learningCovar        = Lambda.asDiagonal();
    _cubes[macroCoordinates]            = CubeModel(learningPrediction,learningCovar);
  }
//--------------------------------------------------------------------------
  bool GPPrediction::predictOutput(const VectorDimInputd & pos, VectorDimOutputd & predictedOutputValue) const
  {
    const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it = _cubes.find(getPosMacroCoordinatesTriplet(pos));
    bool predictionFromModelpossible = checkModelOfCubeExists(it);
    if (predictionFromModelpossible)
    {
      Eigen::MatrixXd PhiStar = computePhiStar(getCenteredPosInCube(pos));
      predictedOutputValue = computeOutputPrediction((it->second).learningPrediction , PhiStar) ;
    }
    return predictionFromModelpossible;
  }
  //--------------------------------------------------------------------------
    bool GPPrediction::predictOutputCovar(const VectorDimInputd & pos, MatrixDimOutputd & predictedOutputCovar) const
    {
      const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it = _cubes.find(getPosMacroCoordinatesTriplet(pos));
      bool predictionFromModelpossible = checkModelOfCubeExists(it);
      if (predictionFromModelpossible)
      {
        Eigen::MatrixXd PhiStar = computePhiStar(getCenteredPosInCube(pos));
        predictedOutputCovar = computeOutputPredictionCovar((it->second).learningCovar , PhiStar) ;
      }
      return predictionFromModelpossible;
    }
  //--------------------------------------------------------------------------
    bool GPPrediction::predictOutput(const VectorDimInputd & pos, VectorDimOutputd & predictedOutputValue, MatrixDimOutputd & predictedOutputCovar) const
    {
      const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it = _cubes.find(getPosMacroCoordinatesTriplet(pos));
      bool predictionFromModelpossible = checkModelOfCubeExists(it);
      if (predictionFromModelpossible)
      {
        const  Eigen::MatrixXd PhiStar = computePhiStar(getCenteredPosInCube(pos));
        predictedOutputValue = computeOutputPrediction((it->second).learningPrediction , PhiStar) ;
        predictedOutputCovar = computeOutputPredictionCovar((it->second).learningCovar , PhiStar) ;
      }
      return predictionFromModelpossible;
    }
  //--------------------------------------------------------------------------
    Eigen::MatrixXd GPPrediction::computePhiStar(const VectorDimInputd & centeredPosInCube) const
    { return GPLearning::PhiBlockLine(centeredPosInCube, _basisOfEigenfunctions, _LxLyLzExtended) ; }
//--------------------------------------------------------------------------
  void GPPrediction::serializeLearningPrediction(std::vector<triplet> & coordinateList, Eigen::VectorXd & serializedLearningPrediction) const
  {
      const unsigned int nbCubes = _cubes.size() ;
      if (nbCubes == 0 && coordinateList.size() != 0)
      {return ;}
      const unsigned int sizePrediction = _cubes.begin()->second.learningPrediction.size() ;

      serializedLearningPrediction = Eigen::VectorXd(sizePrediction*nbCubes) ;
      unsigned int cubeId = 0 ;
      for (std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it =_cubes.begin(); it!=_cubes.end(); ++it)
      {
          coordinateList.push_back(it->first) ;
          serializedLearningPrediction.segment(cubeId*sizePrediction, sizePrediction) = it->second.learningPrediction ;
          cubeId++ ;
      }
      return ;
  }
//--------------------------------------------------------------------------
  void GPPrediction::unserializeLearningPrediction(const std::vector<triplet> & coordinateList, const Eigen::VectorXd & serializedLearningPrediction)
  {
      const unsigned int nbCubes = coordinateList.size() ;
      const unsigned int sizePrediction = serializedLearningPrediction.size()/nbCubes ;

      for (unsigned int i = 0; i < nbCubes; i++)
      { _cubes.at(coordinateList.at(i)).learningPrediction = serializedLearningPrediction.segment(i*sizePrediction, sizePrediction) ; }
      return ;
  }
//--------------------------------------------------------------------------
  void GPPrediction::writeTextGPPrediction(const std::string & fileName, const unsigned int precision) const
  {
    std::ofstream outfile(fileName);
    outfile.precision(precision);
    outfile << std::fixed ;
    outfile << "# Map Model: half-dimensions (dim/2) of cubes x, y, z, half-dimensions of extended cubes for model x, y, z, starting Position x,y,z, sigmaLinSquare, sigmaSESquare, lSESquare, sigmaNoiseSquareModel, nb Eigenfunctions, nb Cubes" << '\n';

    for (unsigned int i = 0; i<dimInput; i++)
    {
      outfile << _LxLyLz(i) << " ";
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      outfile << _LxLyLzData(i) << " ";
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      outfile << _LxLyLzExtended(i) << " ";
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      outfile << _pos0(i) << " ";
    }

    outfile << _sigmaLinSquare << " " << _sigmaSESquare << " " << _lSESquare << " " << _sigmaNoiseSquareModel << " ";
    const unsigned int nbEigenfunctions = _basisOfEigenfunctions.size();
    const unsigned int nbCubes = _cubes.size();
    outfile  << nbEigenfunctions << " " << nbCubes << '\n';
    for (unsigned int i = 0; i<nbEigenfunctions;i++)
      {
	       VectorDimInputI eigenfunction = _basisOfEigenfunctions.at(i);
         for (unsigned int j = 0; j<dimInput; j++)
         {
	        outfile << eigenfunction(j) << " ";
          }
          outfile << '\n';
      }

    for (std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it =_cubes.begin(); it!=_cubes.end(); ++it)
      {
	triplet cubeCoordinate = it->first;
for (unsigned int j = 0; j<dimInput; j++)
{
	outfile << (cubeCoordinate)(j) << " ";
}
outfile << std::endl;
	CubeModel predictionCube = it->second;
	Eigen::VectorXd cubeLearningPrediction = predictionCube.learningPrediction;
	Eigen::MatrixXd cubeLearningCovar = predictionCube.learningCovar;
	for (unsigned int m = 0; m<dimInput+nbEigenfunctions; m++)
	  {
	    outfile << cubeLearningPrediction(m) << " ";
	  }
	outfile << '\n';
	for (unsigned int m = 0; m<+dimInput+nbEigenfunctions; m++)
	  {
	    for (unsigned int n = m; n<dimInput+nbEigenfunctions; n++)
	      {
		outfile << cubeLearningCovar(m,n) << " ";
	      }
	    outfile << '\n';
	  }
      }
    outfile.close();
  }
//--------------------------------------------------------------------------
  void GPPrediction::readGPPredictionFromFile_impl(const std::string & fileName,
                                                 std::vector<VectorDimInputI> & basisOfEigenfunctions,
                                                 VectorDimInputd & LxLyLz,
                                                 VectorDimInputd & LxLyLzData,
                                                 VectorDimInputd & LxLyLzExtended,
                                                 VectorDimInputd & pos0,
                                                 double & sigmaLinSquare, double & sigmaSESquare,
                                                 double & lSESquare, double & sigmaNoiseSquareModel,
                                                 std::map<triplet,CubeModel,ComparisonTriplet> & mapCubesModel)
  {
      std::ifstream inFile;
      inFile.open(fileName);
      if (!inFile)
      {
          std::cerr << "Unable to open file";
          exit(1); // terminate with error
      }

      std::string line ;
      std::getline (inFile,line); // remove comment

      for (unsigned int i = 0; i<dimInput; i++)
      {
        inFile >> LxLyLz(i);
      }
      for (unsigned int i = 0; i<dimInput; i++)
      {
        inFile >> LxLyLzData(i);
      }
      for (unsigned int i = 0; i<dimInput; i++)
      {
        inFile >> LxLyLzExtended(i);
      }
      for (unsigned int i = 0; i<dimInput; i++)
      {
        inFile >> pos0(i);
      }

      unsigned int nbEigenfunctions;
      unsigned int nbCubes;

      inFile >> sigmaLinSquare >> sigmaSESquare >> lSESquare >> sigmaNoiseSquareModel;

      inFile >> nbEigenfunctions >> nbCubes; //read values
      for (unsigned int i = 0; i<nbEigenfunctions; i++)
      {
          VectorDimInputI eigenfunction;
          for (unsigned int j = 0; j<dimInput; j++)
          {
            inFile >> eigenfunction(j); //read values
          }

          basisOfEigenfunctions.push_back(eigenfunction);
      }
      // build mapCubesModel
      for (unsigned int i = 0; i<nbCubes; i++)
      {
        triplet cubeCoordinateTriplet;
        for (unsigned int j = 0; j<dimInput; j++)
        {
          inFile >> cubeCoordinateTriplet(j); //read values
        }

          Eigen::VectorXd cubeLearningPrediction(dimInput+nbEigenfunctions);
          for (unsigned int m = 0; m<dimInput+nbEigenfunctions; m++)
          { inFile >> cubeLearningPrediction(m); }

          Eigen::MatrixXd cubeLearningCovar(dimInput+nbEigenfunctions,dimInput+nbEigenfunctions);
          for (unsigned int m = 0; m<dimInput+nbEigenfunctions; m++)
          {
              for (unsigned int n = m; n<dimInput+nbEigenfunctions; n++)
              { inFile >> cubeLearningCovar(m,n);
                  // because covariance matrix is symmetric
                  cubeLearningCovar(n,m) = cubeLearningCovar(m,n); }
          }

          mapCubesModel[cubeCoordinateTriplet] = CubeModel(cubeLearningPrediction,cubeLearningCovar);
      }

      inFile.close() ;

      return ;
  }
//--------------------------------------------------------------------------
  GPPrediction GPPrediction::readGPPredictionFromFile(const std::string & fileName)
  {
      std::vector <VectorDimInputI> basisOfEigenfunctions;
      VectorDimInputd LxLyLz;
      VectorDimInputd LxLyLzData;
      VectorDimInputd LxLyLzExtended;
      VectorDimInputd pos0;
      double sigmaLinSquare;
      double sigmaSESquare;
      double lSESquare;
      double sigmaNoiseSquareModel;
      std::map<triplet,CubeModel,ComparisonTriplet> mapCubesModel;
      readGPPredictionFromFile_impl(fileName, basisOfEigenfunctions,
                                   LxLyLz, LxLyLzData, LxLyLzExtended, pos0,
                                   sigmaLinSquare, sigmaSESquare, lSESquare, sigmaNoiseSquareModel,
                                   mapCubesModel) ;
      return (GPPrediction(basisOfEigenfunctions, LxLyLz, LxLyLzData, LxLyLzExtended, pos0, sigmaLinSquare, sigmaSESquare, lSESquare, sigmaNoiseSquareModel, mapCubesModel)) ;
  }
//--------------------------------------------------------------------------
  std::unique_ptr<GPPrediction> GPPrediction::readGPPredictionFromFile_ptr(const std::string & fileName)
  {
      std::vector <VectorDimInputI> basisOfEigenfunctions;
      VectorDimInputd LxLyLz;
      VectorDimInputd LxLyLzData;
      VectorDimInputd LxLyLzExtended;
      VectorDimInputd pos0;
      double sigmaLinSquare;
      double sigmaSESquare;
      double lSESquare;
      double sigmaNoiseSquareModel;
      std::map<triplet, CubeModel,ComparisonTriplet> mapCubesModel;
      readGPPredictionFromFile_impl(fileName, basisOfEigenfunctions,
                                   LxLyLz, LxLyLzData, LxLyLzExtended, pos0,
                                   sigmaLinSquare, sigmaSESquare, lSESquare, sigmaNoiseSquareModel,
                                   mapCubesModel) ;
      return (std::unique_ptr<GPPrediction>(new GPPrediction(basisOfEigenfunctions, LxLyLz, LxLyLzData, LxLyLzExtended, pos0, sigmaLinSquare, sigmaSESquare, lSESquare, sigmaNoiseSquareModel, mapCubesModel))) ;
  }
//--------------------------------------------------------------------------
  VectorDimInputd GPPrediction::readGPPredictionOffsetFromFile(const std::string & fileName)
  {
    std::ifstream inFile;
    inFile.open(fileName);
    if (!inFile)
      {
	std::cerr << "Unable to open file";
	exit(1); // terminate with error
      }

    std::string line ;
    std::getline (inFile,line); // remove comment

    VectorDimInputd LxLyLz;
    VectorDimInputd LxLyLzData;
    VectorDimInputd LxLyLzExtended;
    VectorDimInputd pos0;

    for (unsigned int i = 0; i<dimInput; i++)
    {
      inFile >> LxLyLz(i);
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      inFile >> LxLyLzData(i);
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      inFile >> LxLyLzExtended(i);
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      inFile >> pos0(i);
    }

    inFile.close() ;

    return pos0 ;
}

//--------------------------------------------------------------------------
indexInCubeMap GPPrediction::separateCubes(const std::vector<VectorDimInputd> & pos) const
{
indexInCubeMap allCubes;
const unsigned int nbDatapoints = pos.size();
for (unsigned int indexOfData = 0; indexOfData < nbDatapoints ; indexOfData++)
{
  std::vector<triplet> listOfCoordinatesOfCubesContainingData = getPossibleCoordinates(pos.at(indexOfData));
  for (std::vector<triplet>::iterator macroCoordIt = listOfCoordinatesOfCubesContainingData.begin(); macroCoordIt!=listOfCoordinatesOfCubesContainingData.end(); macroCoordIt++)
  {
    triplet cubeMacroCoord = *macroCoordIt;
    indexInCubeMap::iterator it = allCubes.find(cubeMacroCoord);
    if (it == allCubes.end())
    {
      std::vector<unsigned int> dataIndexInCube;
      dataIndexInCube.push_back(indexOfData);
      allCubes[cubeMacroCoord] = dataIndexInCube;
    }
    else
    {
      (it->second).push_back(indexOfData);
    }
  }
}
return allCubes;
}


//--------------------------------------------------------------------------
  std::vector<triplet> GPPrediction::getPossibleCoordinates(const VectorDimInputd & pos) const
  {
    const VectorDimInputd delta = 2*(_LxLyLzData - _LxLyLz);
    const VectorDimInputd macroCoordinate = getPosMacroCoordinates(pos);
    const VectorDimInputd posInCube = getCenteredPosInCube(pos);
    std::vector<std::vector<int>> macroCoordinateByCoordinate(dimInput);
    for (unsigned int i = 0; i < dimInput; i++)
    {
      macroCoordinateByCoordinate[i].push_back(macroCoordinate(i));
      if (std::abs(posInCube(i)) > (_LxLyLz(i) - delta(i)))
      {
        macroCoordinateByCoordinate[i].push_back(macroCoordinate(i) + 1 - 2*std::signbit(posInCube(i)));
      }
    }

    std::vector<triplet> listOfCoordinatesOfCubesContainingData;
    unsigned nbOfCubesContainingData = 1;
    for (unsigned int i = 0; i < dimInput; i++)
    {
        nbOfCubesContainingData *= macroCoordinateByCoordinate[i].size();
    }
    for (unsigned int i = 0; i < nbOfCubesContainingData; i++)
    {
      triplet newCubeContainingData ;
      unsigned int indexTmp = 0;
      unsigned int coordOfIthValue = i;
      while (indexTmp < dimInput)
      {
        newCubeContainingData(indexTmp) = macroCoordinateByCoordinate[indexTmp].at(coordOfIthValue% macroCoordinateByCoordinate[indexTmp].size());
        coordOfIthValue = coordOfIthValue/ macroCoordinateByCoordinate[indexTmp].size();
        indexTmp += 1;
      }
        listOfCoordinatesOfCubesContainingData.push_back(newCubeContainingData );
    }
    return listOfCoordinatesOfCubesContainingData;
  }
  //--------------------------------------------------------------------------
