/**
 * @file GPPrediction.h
 *
 * @brief
 * GPPrediction
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 *
 *
 * @date
 * Modified on 2020/09/26
 *
 */

////////////////////////////////////////////////////////////////////////////////

#ifndef GP_PREDICTION_H
#define GP_PREDICTION_H

#pragma once

#include <iostream>
#include <map>
#include <memory>
#include <fstream>
#include <cmath>
#include "GPLearning.h"
#include "GPTypes.h"

  class GPPrediction
  {
  private:

    typedef struct CubeModel
    {
      Eigen::VectorXd learningPrediction;
      Eigen::MatrixXd learningCovar;
        CubeModel(){}
        CubeModel(const Eigen::VectorXd & a_learningPrediction,
                  const Eigen::MatrixXd & a_learningCovar)
        : learningPrediction( a_learningPrediction)
        , learningCovar( a_learningCovar){}
    } CubeModel ;

  private :
    // all hyperparameters
    const double _sigmaLinSquare ;
    const double _sigmaSESquare ;
    const double _lSESquare ;
    const double _sigmaNoiseSquareModel ;

    // set of eigen fonction used for the map
    const std::vector<VectorDimInputI> _basisOfEigenfunctions;

    // dimension of cube to cut the trajectory into pieces
    const VectorDimInputd _LxLyLz;

    // dimension of overlapping cube used for learning (to avoid prediction discontinuity)
    const VectorDimInputd _LxLyLzData;

    // dimension of cube used for extrapolation (to avoid dirichlet limit)
    const VectorDimInputd _LxLyLzExtended;

    // starting position of the map to interprete macro coordinate in CubeModels
    const VectorDimInputd _pos0;

    // If true, the map will apply first position offset.
    bool _apply_pos0_offset ;

    // all cube data model needed for data extrapolation
    std::map<triplet,CubeModel,ComparisonTriplet> _cubes;

  private :
    GPPrediction(const std::vector<VectorDimInputI> & basisOfEigenfunctions,
    const VectorDimInputd & LxLyLz,
    const VectorDimInputd & LxLyLzData,
    const VectorDimInputd & LxLyLzExtended,
    const VectorDimInputd & pos0,
    const double sigmaLinSquare, const double sigmaSESquare,
    const double lSESquare, const double sigmaNoiseSquareModel,
    const std::map<triplet,CubeModel,ComparisonTriplet> & cubes);

    static void readGPPredictionFromFile_impl(const std::string & fileName,
                                             std::vector<VectorDimInputI> & basisOfEigenfunctions,
                                             VectorDimInputd & LxLyLz,
                                             VectorDimInputd & LxLyLzData,
                                             VectorDimInputd & LxLyLzExtended,
                                             VectorDimInputd & pos0,
                                             double & sigmaLinSquare, double & sigmaSESquare,
                                             double & lSESquare, double & sigmaNoiseSquareModel,
                                             std::map<triplet,CubeModel,ComparisonTriplet> & mapCubesModel) ;

  public :
    GPPrediction(const VectorDimInputI & nbFrequsForBasisOfEigenfunctions,
    const VectorDimInputd & dimCubes,
    const VectorDimInputd & pos0,
    const double sigmaLinSquare, const double sigmaSESquare,
    const double lSESquare, const double sigmaNoiseSquareModel);

    // read map in a file
    static GPPrediction                  readGPPredictionFromFile    (const std::string & fileName);
    static std::unique_ptr<GPPrediction> readGPPredictionFromFile_ptr(const std::string & fileName);

    // read map position offset in a file
    static VectorDimInputd readGPPredictionOffsetFromFile(const std::string & fileName);

    // write map to a readable file
    void writeTextGPPrediction(const std::string & fileName, const unsigned int precision = 6) const ;


    void update(const posValueVect & pos_and_value_W);

    void update(const posCovValueVect & pos_cov_p_value_W);

    void update(const VectorDimInputd & pos, const VectorDimOutputd & value_W);

    void update(const VectorDimInputd & pos, const MatrixDimInputd cov_p, const VectorDimOutputd & value_W);

    bool predictOutput(const VectorDimInputd & pos, VectorDimOutputd & predictedOutputValue) const ;

    bool predictOutputCovar(const VectorDimInputd & pos, MatrixDimOutputd & predictedOutputCovar) const ;

    bool predictOutput(const VectorDimInputd & pos, VectorDimOutputd & predictedOutputValue, MatrixDimOutputd & predictedOutputCovar) const ;

    inline void setApply_pos0_offset(const bool apply_pos0_offset ) { _apply_pos0_offset = apply_pos0_offset ;}

    inline VectorDimInputd get_pos0() const { return _pos0 ;}

    // Compute and print performance of the map prediction over a list of localized raw data //
    void mapPredictionPerformance(const std::string & fileName, const PairPosValue & pose_WValueReloc_from_Imu_and_value_raw) const ;

    void serializeLearningPrediction(std::vector<triplet> & coordinateList, Eigen::VectorXd & serializedLearningPrediction) const ;

    void unserializeLearningPrediction(const std::vector<triplet> & coordinateList, const Eigen::VectorXd & serializedLearningPrediction) ;


  private :
    Eigen::MatrixXd computePhiStar(const VectorDimInputd & centeredPosInCube) const ;

    inline VectorDimOutputd computeOutputPrediction(const Eigen::VectorXd & learningPrediction, const Eigen::MatrixXd & PhiStar) const
    { return PhiStar*learningPrediction ;}

    inline MatrixDimOutputd computeOutputPredictionCovar(const Eigen::MatrixXd & learningCovar, const Eigen::MatrixXd & PhiStar) const
    { return PhiStar*learningCovar*PhiStar.transpose() ;}

    void updateCube(const triplet & macroCoordinates, const posValueVect & listCenteredPosInCubeAndValue_W);
    void updateCube(const triplet & macroCoordinates, const posCovValueVect & listCenteredPosInCubePosCovAndValue_W);


    void initCube(const triplet & macroCoordinates);

    // repartition of all (position, measures) in the corresponding cubes that use this data to create their cubes models
   indexInCubeMap separateCubes(const std::vector<VectorDimInputd> & pos) const ;

    // gives the macrocoordinate (for one dimension) of all the cubes that contain a data of position for any number of dimensions macroCoordinateDecimal1d
    std::vector<triplet> getPossibleCoordinates(const VectorDimInputd & pos) const;

    inline VectorDimInputd getPosMacroCoordinates(const VectorDimInputd & pos) const
    {
      VectorDimInputd pos0_offset = ( (_apply_pos0_offset) ? _pos0 : VectorDimInputd::Zero() )   ;
      VectorDimInputd p_centered = (pos - pos0_offset).cwiseProduct((2*_LxLyLz).cwiseInverse());
      VectorDimInputd p_centered_round;
      for (unsigned int i = 0; i<dimInput; i++)
      {
        p_centered_round(i) = std::round(p_centered(i));
      }
      return p_centered_round;
    }

    inline triplet getPosMacroCoordinatesTriplet(const VectorDimInputd & pos) const
    {
      VectorDimInputd posMacroCoordinates = getPosMacroCoordinates(pos) ;
      triplet posMacroCoordinatesTriplet;
      for (unsigned int i = 0; i<dimInput; i++)
      {
        posMacroCoordinatesTriplet(i) = posMacroCoordinates(i);
      }
      return posMacroCoordinatesTriplet;
    }

    inline VectorDimInputd getCenteredPosInCube(const VectorDimInputd & pos) const
    {
      VectorDimInputd pos0_offset = ( (_apply_pos0_offset) ? _pos0 : VectorDimInputd::Zero() )   ;
      return ( pos - (pos0_offset + (2*_LxLyLz.cwiseProduct(getPosMacroCoordinates(pos))) ) ) ;
    }

    inline VectorDimInputd getCenteredPosInCube(const VectorDimInputd & pos, const VectorDimInputd macroCoord) const
    {
      VectorDimInputd pos0_offset = ( (_apply_pos0_offset) ? _pos0 : VectorDimInputd::Zero() );
      return ( pos - (pos0_offset + (2*_LxLyLz.cwiseProduct(macroCoord)) ) ) ;
    }

    inline VectorDimInputd convertTripletInVectorDimInputd (const triplet & tripletCoord) const
    {
      VectorDimInputd vectorDimInputdCoord;
      for (unsigned int i = 0; i<dimInput; i++)
      {    vectorDimInputdCoord(i) = tripletCoord(i);  }
      return vectorDimInputdCoord;
    }

    inline VectorDimInputd getCenteredPosInCube(const VectorDimInputd & pos, const triplet & macroCoord) const
    {
      return getCenteredPosInCube(pos, convertTripletInVectorDimInputd(macroCoord));
    }

    inline posValueStruct getCenteredPosInCube(const posValueStruct & posValueIn, const triplet & macroCoord) const
    {return posValueStruct(getCenteredPosInCube(posValueIn.position, convertTripletInVectorDimInputd(macroCoord)), posValueIn.value);}

    inline posCovValueStruct getCenteredPosInCube(const posCovValueStruct & posCovValueIn, const triplet & macroCoord) const
    {return posCovValueStruct(getCenteredPosInCube(posCovValueIn.position, convertTripletInVectorDimInputd(macroCoord)), posCovValueIn.covPos, posCovValueIn.value);}

    inline bool checkModelOfCubeExists(const std::map<triplet,CubeModel,ComparisonTriplet>::const_iterator it) const
    { return (it != _cubes.end()) ; }

  } ;



#endif
