/**
 * @file GPTypes.cpp
 *
 * @brief
 * List of types used for Gaussian Processes, with inputs and outputFileErrLocSimData
 * for multidimensional inputs, l_SE is the same.
 * output dimension should be equal to 1
 * All Matrices and Vectors operations are based on the Eigen library
 *
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 * @date
 * Modified on 2023/04/26
 */

#include "GPTypes.h"


  posValueStruct::posValueStruct (VectorDimInputd position, VectorDimOutputd value)
  : position(position)
  , value(value)
  {
    if (testValueData(value))
    {
      throw std::runtime_error("vector seems wrong");
    }
  }

  posCovValueStruct::posCovValueStruct (VectorDimInputd position, MatrixDimInputd covPos, VectorDimOutputd value)
  : position(position)
  , covPos(covPos)
  , value(value)
  {
    if (testValueData(value))
    {
      throw std::runtime_error("vector seems wrong");
    }
  }

  bool testValueData (VectorDimOutputd value)
  // return true if there is a problem with the value data
  // they depend on the max value expected
  {
    return ( value.norm() > 200);
  }

  void separateData (const posCovValueVect& dataIn, std::vector<VectorDimInputd>& positionOut, std::vector<VectorDimOutputd>& valueOut, std::vector<MatrixDimInputd>& positionCovOut)
  {
    int nbDatapoints = dataIn.size();
    positionOut.resize(nbDatapoints);
    valueOut.resize(nbDatapoints);
    positionCovOut.resize(nbDatapoints);
    for (int i = 0; i<nbDatapoints; i++)
    {
      positionOut.at(i) = dataIn.at(i).position;
      valueOut.at(i) = dataIn.at(i).value;
      positionCovOut.at(i) = dataIn.at(i).covPos;
    }
  }

  void separateData (const posValueVect& dataIn, std::vector<VectorDimInputd>& positionOut, std::vector<VectorDimOutputd>& valueOut)
  {
    int nbDatapoints = dataIn.size();
    positionOut.resize(nbDatapoints);
    valueOut.resize(nbDatapoints);
    for (int i = 0; i<nbDatapoints; i++)
    {
      positionOut.at(i) = dataIn.at(i).position;
      valueOut.at(i) = dataIn.at(i).value;
    }
  }


  struct posValue
  {
    VectorDimInputd position;
    VectorDimOutputd value;
  };
