/**
 * @file GPTypes.h
 *
 * @brief
 * List of types used for Gaussian Processes, with inputs and outputFileErrLocSimData
 * for multidimensional inputs, l_SE is the same.
 * output dimension should be equal to 1
 * All Matrices and Vectors operations are based on the Eigen library
 *
 * Code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023
 * @date
 * Modified on 2023/04/26
 */

////////////////////////////////////////////////////////////////////////////////

#ifndef GP_TYPES_H
#define GP_TYPES_H

#pragma once

#include <map>
#include <Eigen/Dense>

const unsigned int dimInput = 1;
const unsigned int dimOutput = 1;

typedef Eigen::Matrix<int, dimInput, 1>    VectorDimInputI       ;
typedef Eigen::Matrix<double, dimInput, 1>    VectorDimInputd       ;
typedef Eigen::Matrix<double, dimInput, dimInput>    MatrixDimInputd       ;
typedef Eigen::Matrix<double, dimOutput, 1>    VectorDimOutputd       ;
typedef Eigen::Matrix<double, dimOutput, dimOutput>    MatrixDimOutputd       ;

constexpr double CST_PI = 3.1415926535897932384626433832795;
constexpr double CST_RadToDeg = (180./CST_PI);
//--------------------------------------------------------------------------

  using triplet   = VectorDimInputI;// size of input



struct ComparisonTriplet
{
  bool operator() (const triplet triplet1, const triplet triplet2) const
  {
    const unsigned int nbDims = triplet1.size();
    unsigned int indexTmp = 0;
    while (indexTmp < nbDims &&  triplet1(indexTmp) == triplet2(indexTmp))
    {
      indexTmp += 1;
    };
    if (indexTmp == nbDims)
    {
      // the two values are equal
      return false;
    }
    else
    {
      return triplet1(indexTmp) < triplet2(indexTmp);
    };
  };
};

  using PairPosValue = std::vector <std::pair<VectorDimInputd,VectorDimOutputd> > ;

  struct posValueStruct
  {
    const VectorDimInputd position;
    const VectorDimOutputd value;

    posValueStruct(VectorDimInputd position, VectorDimOutputd value);
  };
  struct posCovValueStruct
  {
    const VectorDimInputd position;
    const MatrixDimInputd covPos;
    const VectorDimOutputd value;

    posCovValueStruct (VectorDimInputd position, MatrixDimInputd covPos, VectorDimOutputd value);
  };
  using posValueVect = std::vector<posValueStruct>;
  using posCovValueVect = std::vector<posCovValueStruct>;
  using indexInCubeMap = std::map<triplet,std::vector<unsigned int>,ComparisonTriplet>;

  bool testValueData (VectorDimOutputd value);
  void separateData (const posCovValueVect& dataIn, std::vector<VectorDimInputd>& positionOut, std::vector<VectorDimOutputd>& valueOut, std::vector<MatrixDimInputd>& positionCovOut);
  void separateData (const posValueVect& dataIn, std::vector<VectorDimInputd>& positionOut, std::vector<VectorDimOutputd>& valueOut);



  void separatePairVec(const posValueVect & pairIn, std::vector<VectorDimInputd>& firstOut, std::vector<VectorDimOutputd>& secondOut);


#endif
