/**
 * @file main.cpp
 *
 * @brief
 * Main of the code for the paper Reduced-Rank Online Gaussian Process Modeling
 * With Uncertain Inputs, submitted at Neurips 2023 (C++ model)
 *
 * @date
 * Modified on 2023/02/10
 *
 */


#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <string>
#include <stdlib.h>

#include "GPLearning.h"
#include "GPPrediction.h"
#include "GPTypes.h"
// boost
#include <boost/shared_ptr.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem.hpp>
#include <random>
#include <Eigen/Dense>
namespace po = boost::program_options;
namespace fs = boost::filesystem;

//-----------------------------------------------------------------------------
Eigen::MatrixXd readHyperparams (const std::string dataFile, const unsigned int nbIter)
{
  // suppose startingData = 1 (not 0) is the the first data, and endingData is included
  std::ifstream inFile;
  inFile.open(dataFile);
  if (!inFile)
  {
    throw("Unable to open file");
  }
  std::string line ;
  unsigned int index = 0;
  Eigen::MatrixXd allHyperparams(nbIter, 4);
  //read line by line
  while (!inFile.eof() && index < nbIter+1)
  {
    inFile >> allHyperparams(index,0)>> allHyperparams(index,1)>> allHyperparams(index,2) >> allHyperparams(index,3);
    index++;
  }
  inFile.close();
  return allHyperparams;
}
//----------------------------------------------------------------------------
posCovValueVect readOutputDataWithPoseErrorLoc (const std::string dataFile, const unsigned int startingData, const unsigned int endingData)
{
  // suppose startingData = 1 (not 0) is the the first data, and endingData is included
  // datafile is suposed to be a csv such that each line is :
  // timestamp, input (or pos for position), input cov and measure of the output
  std::ifstream inFile;
  inFile.open(dataFile);
  if (!inFile)
  {
    throw("Unable to open file");
  }
  std::string line ;
  for (int i = 1 ; i < startingData; i++)
  {
    std::getline (inFile,line);
  }

  posCovValueVect posCovAndOutput;
  int index = startingData;
  VectorDimOutputd newOutputData;
  VectorDimInputd newPosData;
  double timestamp;
  MatrixDimInputd covPos;
  //read line by line
  while (!inFile.eof() && index < endingData+1)
  {
    for (unsigned int i = 0; i<dimInput; i++)
    {
      inFile >> newPosData(i);
    }
    for (unsigned int i = 0; i<dimOutput; i++)
    {
      inFile >> newOutputData(i);
    }
    for (unsigned int i = 0; i<dimInput; i++)
    {
      for (unsigned int j = 0; j<= i; j++)
      {
        inFile >> covPos(i,j);
        covPos(j,i) = covPos(i,j); // by symmetry
      }
    }
    posCovValueStruct newPosCovOutputAndData (newPosData, covPos, newOutputData);

    posCovAndOutput.push_back(newPosCovOutputAndData);
    index++;
  }
  inFile.close();
  return posCovAndOutput;
}
//----------------------------------------------------------------------------
int main(int argc, char const *argv[])
{
  std::cout << "****RRONIG****" << std::endl;
// here this code is for 1D but can easily be adapted for any number of dimensions
// (we tried it in 3D and the results were right)
   ////////////////////////////////////////////////////////////////////////////////////
  const unsigned int nbIter = 1; // number of files to be read with data
  const unsigned int nbDatapointsXaxisPred = 100;
  const unsigned int nbSamplesPred = 200; //200
//  const unsigned int nbInducingPoints = 11;
  std::vector<posCovValueStruct>  allPosCovOutputPredCompData;
//   // to be in the center, so cube could be not too large to cover the whole area
  const double x_min = -5;
const VectorDimInputd dimCubes(5*1.2*2); // dimCubes is its total amplitude, equal to : the interval is [-5, 5]
// other neighboring cubes are often created because data on the border are used for several models,
// in order to ensure a smooth transition if several cubes are explored (we talk of cubes but they can be 1D, 3D or of any dimension)
// to be in the center, so cube could be not too large to cover the whole area
    const VectorDimInputd p_SimData(0.0);
const VectorDimInputI nbFrequsForBasisOfEigenfunctionsSimu(21);

//
// // we get the hyperparams file
std::string hyperparamsFile = "/tmp/hyperparams.txt";
Eigen::MatrixXd hyperparams = readHyperparams(hyperparamsFile, nbIter);

for (unsigned int iIter = 0; iIter<nbIter; iIter++)
{
  std::cout << "nbIter" << iIter << std::endl;
  std::cout << hyperparams(iIter,0) << " "<< hyperparams(iIter,1) << " " << hyperparams(iIter,2)<< " " << hyperparams(iIter,3) << std::endl;

     GPPrediction mapSimulatedData_With_err_loc1D(nbFrequsForBasisOfEigenfunctionsSimu,
                                  dimCubes,
                                  p_SimData,
                                  0,
                                  std::pow(hyperparams(iIter,1),2),
                                  std::pow(hyperparams(iIter,0),2),
                                  std::pow(hyperparams(iIter,2),2));

   unsigned int nbDataSimu = 0;

    const std::string fileNameDataSimRaw = "/tmp/test_"+ std::to_string(iIter+1)+ ".csv";
    system( (std::string("tail -n +2 ") + fileNameDataSimRaw + std::string(" | sed 's/,/ /g' > /tmp/inputAndOutputSim.csv")).c_str() ) ;

    const posCovValueVect allSimulatedData = readOutputDataWithPoseErrorLoc("/tmp/inputAndOutputSim.csv", 0, 1000000);

// we get the predictions with a grid of size x_prediction
   const double x_interval = 0.1;

    // trying with _batch
    const unsigned int batchSizeForUpdate = 1;
    posCovValueVect batch;
    std::cout << "hyperparams(iIter,3) "<< hyperparams(iIter,3) << std::endl;
    std::cout << "std::pow(hyperparams(iIter,3),2) "<< std::pow(hyperparams(iIter,3),2) << std::endl;
    for (posCovValueVect::const_iterator it=allSimulatedData.begin(); it!=allSimulatedData.end(); ++it)
    {
      nbDataSimu +=1;
      if (batch.size() == batchSizeForUpdate && nbDataSimu <= nbSamplesPred)
      {
        mapSimulatedData_With_err_loc1D.update(batch);
        batch.clear();
      }
    posCovValueStruct newData(it->position, it->covPos, it->value);
       batch.push_back(newData);
    }


    std::string fileName = "/tmp/results_rronig_"+ std::to_string(iIter+1)+ ".csv";
    std::ofstream outfile(fileName);
    outfile << std::fixed;
    mapSimulatedData_With_err_loc1D.writeTextGPPrediction(std::string("/tmp/map"+ std::to_string(iIter+1)+".Output"));
    for (int i = 0 ; i <= nbDatapointsXaxisPred; i++)
    {
      VectorDimInputd pos (x_min+i*x_interval);
      VectorDimOutputd predictedOutputValue;
      MatrixDimOutputd predictedOutputVar;
       bool prediction = mapSimulatedData_With_err_loc1D.predictOutput(pos,predictedOutputValue, predictedOutputVar);
      outfile << pos(0) << "," << predictedOutputValue(0) << std::scientific;
      outfile << "," << predictedOutputVar(0) << std::endl;
    }
    outfile.close();
}



  return 0 ;
}
