import logging
from logging.handlers import RotatingFileHandler
import os
import sys

def parse_size(size):
    """
    Parse a human-readable size string (e.g., '10MB', '1GB') into the number of bytes.
    This function supports B, KB, MB, GB, and TB units.
    """
    units = {'B': 1, 'KB': 1024, 'MB': 1024**2, 'GB': 1024**3, 'TB': 1024**4}
    size = size.upper()
    if size[-2:] in units:
        return int(float(size[:-2]) * units[size[-2:]])
    elif size[-1:] in units:
        return int(float(size[:-1]) * units[size[-1:]])
    return int(size)

class StreamToLogger:
    """
    Fake file-like stream object that redirects writes to a logger instance.
    It takes in the logger and the logging level as parameters and uses these to log messages
    that are written to this stream.
    """
    def __init__(self, logger, log_level=logging.INFO):
        self.logger = logger
        self.log_level = log_level
        self.linebuf = ''

    def write(self, buf):
        for line in buf.rstrip().splitlines():
            self.logger.log(self.log_level, line.rstrip())

    def flush(self):
        """
        Handle the flush command by doing nothing.
        This is needed to handle the interface requirements of the stream.
        """
        pass

def setup_logger(log_path='../logs', log_size='10MB', backup_count=3, log_level=logging.INFO):
    """
    Set up and return a logger object that logs to both file and console.
    The log file is saved in the specified directory with rotating behavior.
    
    Parameters:
        log_path: Directory to save the log file.
        log_size: Maximum size of log file before rotation in a human-readable format.
        backup_count: Number of rotated log files to keep.
        log_level: Level of logging.
    """
    # Ensure the directory for log_path exists
    os.makedirs(log_path, exist_ok=True)
    log_filename = os.path.join(log_path, "console.log")

    # Create a logger
    logger = logging.getLogger(__name__)
    logger.setLevel(log_level)

    # Calculate the maximum log file size in bytes
    max_size_bytes = parse_size(log_size)
    file_handler = RotatingFileHandler(log_filename, maxBytes=max_size_bytes, backupCount=backup_count)
    
    # Create and set formatter
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    file_handler.setFormatter(formatter)
    
    # Add file handler to the logger
    logger.addHandler(file_handler)

    # Redirect stdout and stderr to logger
    sys.stdout = StreamToLogger(logger, logging.INFO)
    sys.stderr = StreamToLogger(logger, logging.ERROR)

    return logger

# Usage example:
if __name__ == '__main__':
    logger = setup_logger(log_path='../logs', log_size='10MB', backup_count=3, log_level=logging.INFO)
    print("fkkkk")  # This will now be logged as well
