import h5py
import pyarrow as pa
import pyarrow.parquet as pq
import numpy as np
import io

def list_h5_datasets(h5_path: str):
    """Prints all datasets inside an HDF5 file, including their paths and shapes."""
    with h5py.File(h5_path, 'r') as h5_file:
        def visit_datasets(name, obj):
            if isinstance(obj, h5py.Dataset):
                print(f"Dataset: {name}, Shape: {obj.shape}, Dtype: {obj.dtype}")
        
        h5_file.visititems(visit_datasets)

def h5_to_parquet(h5_path: str):
    """Converts specific datasets from an HDF5 file to a Parquet file.
    
    Args:
        h5_path (str): Path to the input HDF5 file.
    """
    # Define output parquet path
    parquet_path = h5_path.rsplit('.', 1)[0] + '.parquet'
    
    # Open the HDF5 file
    with h5py.File(h5_path, 'r') as h5_file:
        dataset_keys = ["test/pde_640-256"]
        data_dict = {}
        
        for key in dataset_keys:
            print(key)
            if key in h5_file:
                data_dict["u"] = h5_file[key][:][:128]
                print(data_dict["u"].shape)
            else:
                print(f"Warning: {key} not found in {h5_path}")
        
        if not data_dict:
            raise ValueError(f"No specified datasets found in {h5_path}")
        
        # Convert numpy arrays to bytes
        for key in data_dict:
            byte_buffers = [io.BytesIO() for _ in data_dict[key]]
            for arr, buffer in zip(data_dict[key], byte_buffers):
                np.save(buffer, arr)
            data_dict[key] = [buffer.getvalue() for buffer in byte_buffers]
        
        # Create a PyArrow table
        table = pa.Table.from_pydict(data_dict)
        
        # Write to Parquet
        pq.write_table(table, parquet_path)
        
    print(f"Converted {h5_path} to {parquet_path} successfully.")

