import os
import yaml

class Config:
    """Recursive, dot-accessible mapping."""
    def __init__(
        self, 
        dict: dict[str, object] | None = None
    ):
        dict = dict or {}
        for key, value in dict.items():
            self.__dict__[key] = self._wrap(value)

    @classmethod
    def _wrap(
            cls, 
            value: object
    ):
        if isinstance(value, dict):
            return cls(value)
        elif isinstance(value, list):
            return [cls._wrap(item) for item in value]
        else:
            return value
    
    @classmethod
    def from_yaml(
        cls,
        path: str
    ):
        with open(path, 'r') as f:
            config_dict = yaml.safe_load(f) or {}
            
        if 'base_config' in config_dict:
            base_path = os.path.join(
                os.path.dirname(path), 
                config_dict.pop('base_config')
            )
            with open(base_path, 'r') as f:
                base_dict = yaml.safe_load(f) or {}

            base_dict.update(config_dict)
            config_dict = base_dict
                
        return cls(config_dict)
    
    def set(
            self, 
            key_path: str,
            value: object
    ):
        head, *tail = key_path.split('.')
        if not tail:
            setattr(self, head, self._wrap(value))
        else:
            self._ensure_sub_config(head).set(".".join(tail), value)

    def _ensure_sub_config(self, key: str):
        current = getattr(self, key, None)
        if current is None:
            current = Config()
            setattr(self, key, current)
        elif not isinstance(current, Config):
            raise TypeError(f"Cannot descend into non-Config object: {key}.")
        return current
    
    def to_dict(self):
        return {
            key: value.to_dict() if hasattr(value, 'to_dict') else value
            for key, value in self.__dict__.items()
        }