import torch.nn as nn

from . import register_architecture
from ..components import get_component

@register_architecture("ResNet")
class ResNet(nn.Module):
    """
    ResNet architecture with dimension parameter.
    
    Args:
        dimension: Dimension for convolution operations (1, 2, or 3)
        in_channels: Number of input channels
        out_channels: Number of output channels
        hidden_channels: Number of hidden channels
        kernel_size: Size of the convolutional kernel
        padding_mode: Padding mode for the convolutional kernel
        norm: Whether to use normalization
        dropout_rate: Dropout rate
        depth: Number of residual blocks
        bias: Whether to include bias in convolutions
        activation: Name of activation function
        **kwargs: Additional keyword arguments
    """
    def __init__(
            self,
            dimension: int,
            in_channels: int,
            out_channels: int,
            hidden_channels: int,
            kernel_size: int = 3,
            padding_mode: str = "circular",
            norm: bool = True,
            dropout_rate: float = 0.0,
            depth: int = 4,
            bias: bool = True,
            activation: str = "gelu",
            **kwargs
    ):
        super().__init__()

        self.lift = get_component("LiftingLayer")(
            dimension=dimension,
            in_channels=in_channels,
            out_channels=hidden_channels,
            bias=bias
        )

        layers = []
        for _ in range(depth):
            layers.append(get_component("ResidualBlock")(
                dimension=dimension,
                in_channels=hidden_channels,
                out_channels=hidden_channels,
                kernel_size=kernel_size,
                padding_mode=padding_mode,
                norm=norm,
                dropout_rate=dropout_rate,
                bias=bias,
                activation=activation,
                **kwargs
            ))
        self.layers = nn.ModuleList(layers)

        self.proj = get_component("ProjectionLayer")(
            dimension=dimension,
            in_channels=hidden_channels,
            out_channels=out_channels,
            bias=bias
        )
        
    def forward(self, x):
        x = self.lift(x)

        for layer in self.layers:
            x = layer(x)

        x = self.proj(x)
        return x
    
    