import torch.nn as nn

from . import register_component, get_component
from .utils import CONV_TYPES

@register_component("DownBlock")
class DownBlock(nn.Module):
    """
    Downsampling block with dimension parameter.
    
    Args:
        dimension: Dimension for convolution operations (1, 2, or 3)
        in_channels: Number of input channels
        out_channels: Number of output channels
        block_class: Class name of the block to use for convolution
        **kwargs: Additional keyword arguments
    """
    def __init__(
            self,
            dimension: int,
            in_channels: int,
            out_channels: int,
            block_class: str = "ResidualBlock",
            **kwargs
    ):
        super().__init__()

        assert dimension in CONV_TYPES, "Dimension must be 1, 2, or 3"
        Conv = CONV_TYPES[dimension]

        self.down = Conv(
            in_channels,
            in_channels,
            kernel_size=3,
            stride=2,
            padding=1,
            padding_mode=kwargs.get('padding_mode', 'circular'),
            bias=kwargs.get('bias', True)
        )
        
        self.conv = get_component(block_class)(
            dimension=dimension,
            in_channels=in_channels,
            out_channels=out_channels,
            **kwargs
        )
    
    def forward(self, x):
        x = self.down(x)
        x = self.conv(x)
        return x