import numpy as np
from scipy.stats import multivariate_normal
from typing import Tuple



def generate_data(z, dx, h, f, e_x_scale=0.0, e_y_scale=0.0, do_confounding=True, confounding_intensity=0.5) -> Tuple[np.array, np.array]:
    """
    Generates data with confounding affecting both x and y, ensuring that the confounding
    on x and y is correlated.
    The data is generated using the following ste:
    1. Generate a confounder for each sample
    2. Generate noise for x and y
    3. Generate x and y using the given functions h and f via the following ste:
        a. Generate x using h(z, e_x)
        b. Generate y using f(x) + (e_y)

    :param z: np.array, input variable z with shape (n_samples, dz)
    :param dx: int, dimensionality of x
    :param h: function, the function to generate x from z
    :param f: function, the function to generate y from x
    :param e_x_scale: float, scale of noise in x generation
    :param e_y_scale: float, scale of noise in y generation
    :param do_confounding: bool, whether to include confounding in data generation
    :param confounding_intensity: float, intensity of the confounding effect
    
    :return: x, y, where x and y are np.arrays of generated data
    """
    n_samples = z.shape[0]

    # Generate a single confounder for each sample to affect both x and y; dimensions (n_samples,)
    # The confounder is generated using a normal distribution with mean 0 and standard deviation 1 and then scaled by confounding_intensity
    confounder = np.random.normal(scale=confounding_intensity, size=(n_samples, 1))

    # Generating noise for x and y; dimensions (n_samples, dx) and (n_samples,)
    e_x = np.random.normal(scale=e_x_scale, size=(n_samples, dx))
    e_y = np.random.normal(scale=e_y_scale, size=(n_samples, 1))
    
    if do_confounding:
        # Get confounder_x by adding confounder to each row of e_x
        confounder_x = e_x + confounder.squeeze()[:, None]  # Broadcasting confounder to match x's shape, i. e. the confounder is added to each row of e_x
        
        # Generate x with confounding via h(z, confounder_x)
        x = h(z, confounder_x)
        
        # Apply the same confounder to y, ensuring correlated confounding effect
        y = f(x) + e_y + confounder  # Using confounder directly since y is 1-dimensional
    else:
        x = h(z, e_x)
        y = f(x) + e_y
    
    return x, y


