import math
import torch
import torch.nn as nn
from torch.nn import init
import functools
from torch.optim import lr_scheduler
import torch.nn.functional as F
import torch.utils.checkpoint as cp
from collections import OrderedDict
from util.pixel_shuffle import PixelShuffle3d


###############################################################################
# Helper Functions
###############################################################################
def get_scheduler(optimizer, config):
    """Return a learning rate scheduler

    Parameters:
        optimizer          -- the optimizer of the network
        opt (option class) -- stores all the experiment flags; needs to be a subclass of BaseOptions．　
                              config['lr_policy'] is the name of learning rate policy: linear | step | plateau | cosine

    For 'linear', we keep the same learning rate for the first config['epochs']epochs
    and linearly decay the rate to zero over the next config['lr_decay_iters'] epochs.
    For other schedulers (step, plateau, and cosine), we use the default PyTorch schedulers.
    See https://pytorch.org/docs/stable/optim.html for more details.
    """
    lr_policy = config['lr_policy']
    if lr_policy == 'linear':
        def lambda_rule(epoch):
            lr_l = 1.0 - max(0, epoch + 0 - config['epochs']) / float(config['epochs_decay'] + 1)
            return lr_l
        scheduler = lr_scheduler.LambdaLR(optimizer, lr_lambda=lambda_rule)
    elif lr_policy == 'step':
        scheduler = lr_scheduler.StepLR(optimizer, step_size=config['lr_decay_iters'], gamma=0.1)
    elif lr_policy == 'plateau':
        scheduler = lr_scheduler.ReduceLROnPlateau(optimizer, mode='min', factor=0.2, threshold=0.01, patience=5)
    elif lr_policy == 'cosine':
        scheduler = lr_scheduler.CosineAnnealingLR(optimizer, T_max=config['epochs'], eta_min=0)
    else:
        return NotImplementedError('learning rate policy [%s] is not implemented', lr_policy)
    return scheduler

def init_weights(net, init_type='normal', init_gain=0.02):
    """Initialize network weights.

    Parameters:
        net (network)   -- network to be initialized
        init_type (str) -- the name of an initialization method: normal | xavier | kaiming | orthogonal
        init_gain (float)    -- scaling factor for normal, xavier and orthogonal.

    We use 'normal' in the original pix2pix and CycleGAN paper. But xavier and kaiming might
    work better for some applications. Feel free to try yourself.
    """
    def init_func(m):  # define the initialization function
        classname = m.__class__.__name__
        if hasattr(m, 'weight') and (classname.find('Conv') != -1 or classname.find('Linear') != -1):
            if init_type == 'normal':
                init.normal_(m.weight.data, 0.0, init_gain)
            elif init_type == 'xavier':
                init.xavier_normal_(m.weight.data, gain=init_gain)
            elif init_type == 'kaiming':
                init.kaiming_normal_(m.weight.data, a=0, mode='fan_in')
            elif init_type == 'orthogonal':
                init.orthogonal_(m.weight.data, gain=init_gain)
            else:
                raise NotImplementedError('initialization method [%s] is not implemented' % init_type)
            if hasattr(m, 'bias') and m.bias is not None:
                init.constant_(m.bias.data, 0.0)
        elif classname.find('BatchNorm3d') != -1:  # BatchNorm Layer's weight is not a matrix; only normal distribution applies.
            init.normal_(m.weight.data, 1.0, init_gain)
            init.constant_(m.bias.data, 0.0)

    print('initialize network with %s' % init_type)
    net.apply(init_func)  # apply the initialization function <init_func>

def init_net(net, init_type='normal', init_gain=0.02, gpu_ids=[]):
    """Initialize a network: 1. register CPU/GPU device (with multi-GPU support); 2. initialize the network weights
    Parameters:
        net (network)      -- the network to be initialized
        init_type (str)    -- the name of an initialization method: normal | xavier | kaiming | orthogonal
        gain (float)       -- scaling factor for normal, xavier and orthogonal.
        gpu_ids (int list) -- which GPUs the network runs on: e.g., 0,1,2

    Return an initialized network.
    """
    if len(gpu_ids) > 0:
        assert(torch.cuda.is_available())
        net.to(gpu_ids[0])
        net = torch.nn.DataParallel(net, gpu_ids)  # multi-GPUs
    init_weights(net, init_type, init_gain=init_gain)
    return net

def define_G(input_nc, output_nc, netG, norm_layer='batchnorm', num_downs=4, ngf=64, upmode='pixelshuffle', init_type='normal', scale_factor=0,  init_gain=0.02, use_sigmoid=False, drop_rate=0, gpu_ids=[]):
    """Create a generator

    Parameters:
        input_nc (int)      -- the number of channels in input images
        output_nc (int)     -- the number of channels in output images
        netG (str)          -- the architecture's name: unet3d
        norm_layer (str)    -- the norm layer 'batchnorm', 'instancenorm' or 'identity'
        num_downs (int)     -- the number of downsampling layers in the unet
        ngf (int)           -- the number of filters in the last convolutional layer
        upmode (str)        -- the upsampling method 'pixelshuffle', 'interpolation' or 'transposedconv'
        init_type (str)     -- the name of our initialization method.
        init_gain (float)   -- scaling factor for normal, xavier and orthogonal.
        use_sigmoid (bool)  -- use Sigmoid or Tanh as the last layer in the network
        drop_rate (flaot)   -- use dropout regularization in the network with drop_rate if drop_rate > 0
        gpu_ids (int list)  -- which GPUs the network runs on: e.g., 0,1,2

    Returns a generator

    Our current implementation provides several types of generators:
        U-Net: [unet3dfull]
        The paper: https://arxiv.org/abs/1606.06650

        Modified U-Net: [unet3d]
        The paper: http://arxiv.org/abs/1505.04597

        DenseNet: [densenet3d]
        The paper: http://arxiv.org/abs/1608.06993

        ResNet: [resnet_6blocks | resnet_2blocks | resnet_9blocks]
        The paper: https://arxiv.org/abs/1703.10593

    The generator has been initialized by <init_net>. It uses RELU for non-linearity.
    """
    net = None
    # Normalization layers
    if norm_layer == 'batchnorm':
        norm_layer = nn.BatchNorm3d
    elif norm_layer == 'instancenorm':
        norm_layer = nn.InstanceNorm3d
    else:
        norm_layer = Identity

    if netG == 'unet3dfull':
        net = UnetGenerator3DFull(input_nc, output_nc, ngf=ngf, use_sigmoid=use_sigmoid, norm_layer=norm_layer, upmode=upmode, drop_rate=drop_rate)
    elif netG == 'unet3d':
        net = UnetGenerator3D(input_nc, output_nc, num_downs=num_downs, ngf=ngf, use_sigmoid=use_sigmoid, norm_layer=norm_layer, upmode=upmode, drop_rate=drop_rate, scale_factor=scale_factor)
    elif netG == 'densenet3d':
        net = DenseNet3D(input_nc, output_nc, use_sigmoid=use_sigmoid, block_config=(16, 16, 16), upmode=upmode, drop_rate=drop_rate, norm_layer=norm_layer)
    elif netG == 'resnet_2blocks':
        net = ResnetGenerator3D(input_nc, output_nc, ngf=ngf, n_blocks=2, norm_layer=norm_layer, use_sigmoid=use_sigmoid, upmode=upmode, drop_rate=drop_rate)
    elif netG == 'resnet_6blocks':
        net = ResnetGenerator3D(input_nc, output_nc, ngf=ngf, n_blocks=6, norm_layer=norm_layer, use_sigmoid=use_sigmoid, upmode=upmode, drop_rate=drop_rate)
    elif netG == 'resnet_9blocks':
        net = ResnetGenerator3D(input_nc, output_nc, ngf=ngf, n_blocks=9, norm_layer=norm_layer, use_sigmoid=use_sigmoid, upmode=upmode, drop_rate=drop_rate)
    elif netG == 'RRDBNet3D_8blocks':
        net = RRDBNet3D(input_nc, output_nc, ngf, 8, num_downs=num_downs, norm_layer=norm_layer if norm_layer != Identity else None, use_sigmoid=use_sigmoid, upmode=upmode)
    else:
        raise NotImplementedError('Generator model name [%s] is not recognized' % netG)
    return init_net(net, init_type, init_gain, gpu_ids)

def define_D(input_nc, netD, ndf=64, n_layers_D=3, init_type='normal', norm_layer='batchnorm', init_gain=0.02, gpu_ids=[]):
    """Create a discriminator

    Parameters:
        input_nc (int)     -- the number of channels in input images
        ndf (int)          -- the number of filters in the last convolutional block
        n_layers_D (int)   -- the number of convolutional downsampling layers
        netD (str)         -- the architecture's name: basic 
        init_type (str)    -- the name of the initialization method.
        init_gain (float)  -- scaling factor for normal, xavier and orthogonal.
        gpu_ids (int list) -- which GPUs the network runs on: e.g., 0,1,2

    Returns a discriminator

    Our current implementation provides three types of discriminators:
        [basic]: 'PatchGAN' classifier described in the original pix2pix paper, but modified to 3D.
        It can classify whether 2^n_layers×2^n_layersx2^n_layers, so 32x32x32, overlapping patches are real or fake.
        Such a patch-level discriminator architecture has fewer parameters
        than a full-image discriminator and can work on arbitrarily-sized images
        in a fully convolutional fashion.

        [n_layers]: With this mode, you can specify the number of conv layers in the discriminator
        with the parameter <n_layers_D> (default=3 as used in [basic] (PatchGAN).)

        [pixel]: 1x1 PixelGAN discriminator can classify whether a pixel is real or not.
        It encourages greater color diversity but has no effect on spatial statistics.

        The paper: https://arxiv.org/abs/1611.07004

    The discriminator has been initialized by <init_net>. It uses Leakly RELU for non-linearity.
    """
    net = None

    # Normalization layers
    if norm_layer == 'batchnorm':
        norm_layer = nn.BatchNorm3d
    elif norm_layer == 'instancenorm':
        norm_layer = nn.InstanceNorm3d
    else:
        norm_layer = Identity

    if netD == 'basic':  # default PatchGAN classifier
        net = NLayerDiscriminator3D(input_nc, ndf=ndf, n_layers=5, norm_layer=norm_layer)
    elif netD == 'n_layers':  # more options
        net = NLayerDiscriminator3D(input_nc, ndf=ndf, n_layers=n_layers_D, norm_layer=norm_layer)
    elif netD == 'pixel':     # classify if each pixel is real or fake
        net = PixelDiscriminator(input_nc, ndf=ndf, norm_layer=norm_layer)
    else:
        raise NotImplementedError('Discriminator model name [%s] is not recognized' % net)
    
    return init_net(net, init_type, init_gain, gpu_ids)

##############################################################################
# Loss Classes
##############################################################################
class GANLoss(nn.Module):
    """Define different GAN objectives.

    The GANLoss class abstracts away the need to create the target label tensor
    that has the same size as the input.
    """

    def __init__(self, gan_mode, target_real_label=1.0, target_fake_label=0.0):
        """ Initialize the GANLoss class.

        Parameters:
            gan_mode (str) - - the type of GAN objective. It currently supports vanilla, lsgan, and wgangp.
            target_real_label (bool) - - label for a real image
            target_fake_label (bool) - - label of a fake image

        Note: Do not use sigmoid as the last layer of Discriminator.
        LSGAN needs no sigmoid. vanilla GANs will handle it with BCEWithLogitsLoss.
        """
        super(GANLoss, self).__init__()
        self.register_buffer('real_label', torch.tensor(target_real_label))
        self.register_buffer('fake_label', torch.tensor(target_fake_label))
        self.gan_mode = gan_mode
        if gan_mode == 'lsgan':
            self.loss = nn.MSELoss()
        elif gan_mode == 'vanilla':
            self.loss = nn.BCEWithLogitsLoss()
        elif gan_mode in ['wgangp']:
            self.loss = None
        else:
            raise NotImplementedError('gan mode %s not implemented' % gan_mode)

    def get_target_tensor(self, prediction, target_is_real):
        """Create label tensors with the same size as the input.

        Parameters:
            prediction (tensor) - - tpyically the prediction from a discriminator
            target_is_real (bool) - - if the ground truth label is for real images or fake images

        Returns:
            A label tensor filled with ground truth label, and with the size of the input
        """

        if target_is_real:
            target_tensor = self.real_label
        else:
            target_tensor = self.fake_label
        return target_tensor.expand_as(prediction)

    def __call__(self, prediction, target_is_real):
        """Calculate loss given Discriminator's output and grount truth labels.

        Parameters:
            prediction (tensor) - - tpyically the prediction output from a discriminator
            target_is_real (bool) - - if the ground truth label is for real images or fake images

        Returns:
            the calculated loss.
        """
        if self.gan_mode in ['lsgan', 'vanilla']:
            target_tensor = self.get_target_tensor(prediction, target_is_real)
            loss = self.loss(prediction, target_tensor)
        elif self.gan_mode == 'wgangp':
            if target_is_real:
                loss = -prediction.mean()
            else:
                loss = prediction.mean()
        return loss


def cal_gradient_penalty(netD, real_data, fake_data, device, type='mixed', constant=1.0, lambda_gp=10.0):
    """Calculate the gradient penalty loss, used in WGAN-GP paper https://arxiv.org/abs/1704.00028

    Arguments:
        netD (network)              -- discriminator network
        real_data (tensor array)    -- real images
        fake_data (tensor array)    -- generated images from the generator
        device (str)                -- GPU / CPU: from torch.device('cuda:{}'.format(self.gpu_ids[0])) if self.gpu_ids else torch.device('cpu')
        type (str)                  -- if we mix real and fake data or not [real | fake | mixed].
        constant (float)            -- the constant used in formula ( | |gradient||_2 - constant)^2
        lambda_gp (float)           -- weight for this loss

    Returns the gradient penalty loss
    """
    if lambda_gp > 0.0:
        if type == 'real':   # either use real images, fake images, or a linear interpolation of two.
            interpolatesv = real_data
        elif type == 'fake':
            interpolatesv = fake_data
        elif type == 'mixed':
            alpha = torch.rand(real_data.shape[0], 1)
            alpha = alpha.expand(real_data.shape[0], real_data.nelement() // real_data.shape[0]).contiguous().view(*real_data.shape)
            alpha = alpha.to(device)
            interpolatesv = alpha * real_data + ((1 - alpha) * fake_data)
        else:
            raise NotImplementedError('{} not implemented'.format(type))
        interpolatesv.requires_grad_(True)
        disc_interpolates = netD(interpolatesv)
        gradients = torch.autograd.grad(outputs=disc_interpolates, inputs=interpolatesv,
                                        grad_outputs=torch.ones(disc_interpolates.size()).to(device),
                                        create_graph=True, retain_graph=True, only_inputs=True)
        gradients = gradients[0].view(real_data.size(0), -1)  # flat the data
        gradient_penalty = (((gradients + 1e-16).norm(2, dim=1) - constant) ** 2).mean() * lambda_gp        # added eps
        return gradient_penalty, gradients
    else:
        return 0.0, None

##############################################################################
# Generator Network Architectures
##############################################################################

##############################################################################
# Unet 3D Cicek16
##############################################################################

class UnetGenerator3DFull(nn.Module):
    """Create Unet 3D Generator"""
    def __init__(self, input_nc =  1, output_nc = 1, ngf=16, upmode='transposedconv', norm_layer=nn.BatchNorm3d, drop_rate=0, use_sigmoid=False):
        """Construct a Unet generator
        Parameters:
            input_nc (int):         -- the number of channels in input images
            output_nc (int):        -- the number of channels in output images
            ngf (int):              -- number of channels in the first layer
            upmode (str):           -- upsampling mode transposedconv | interpolation | pixelshuffle
            norm_layer (object):    -- normalization layer
            drop_rate (float):      -- use dropout layers if drop_rate > 0
            use_sigmoid (bool):     -- use sigmoid or tanh as last activation funnction
        """
        super(UnetGenerator3DFull,self).__init__()

        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        self.c1 = nn.Sequential(
                    nn.Conv3d(input_nc, ngf//2, 3, padding=1),
                    norm_layer(ngf//2),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf//2, ngf,3, padding=1),
                    norm_layer(ngf),
                    nn.ReLU(inplace=True)
                    )
        self.d1 = nn.Sequential(
                    nn.Conv3d(ngf, ngf, kernel_size=4, stride=2, padding=1, bias=use_bias),
                    norm_layer(ngf),
                    nn.ReLU(inplace=True)
                    )

        self.c2 = nn.Sequential(
                    nn.Conv3d(ngf, ngf,3, padding=1),
                    norm_layer(ngf),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf, ngf*2,3, padding=1),
                    norm_layer(ngf*2),
                    nn.ReLU(inplace=True)
                    )
        self.d2 = nn.Sequential(
                    nn.Conv3d(ngf*2, ngf*2, kernel_size=4, stride=2, padding=1, bias=use_bias),
                    norm_layer(ngf*2),
                    nn.ReLU(inplace=True)
                    )

        self.c3 = nn.Sequential(
                    nn.Conv3d(ngf*2, ngf*2,3, padding=1),
                    norm_layer(ngf*2),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf*2, ngf*4,3, padding=1),
                    norm_layer(ngf*4),
                    nn.ReLU(inplace=True)
                    )
        self.d3 = nn.Sequential(
                    nn.Conv3d(ngf*4, ngf*4, kernel_size=4, stride=2, padding=1, bias=use_bias),
                    norm_layer(ngf*4),
                    nn.ReLU(inplace=True)
                    )


        self.c4 = nn.Sequential(
                    nn.Conv3d(ngf*4, ngf*4,3, padding=1),
                    norm_layer(ngf*4),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf*4, ngf*8,3, padding=1),
                    norm_layer(ngf*8),
                    nn.ReLU(inplace=True)
                    )

        self.u5 = nn.Sequential(get_up_layer(ngf*8, ngf*8, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), nn.Dropout(drop_rate)) if drop_rate > 0 else get_up_layer(ngf*8, ngf*8, mode=upmode, norm_layer=norm_layer, use_bias=use_bias)
        self.c6 = nn.Sequential(
                    nn.Conv3d(ngf*8 + ngf*4, ngf*4,3, padding=1),
                    norm_layer(ngf*4),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf*4, ngf*4,3, padding=1),
                    norm_layer(ngf*4),
                    nn.ReLU(inplace=True)
                    )

        self.u7 = nn.Sequential(get_up_layer(ngf*4, ngf*4, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), nn.Dropout(drop_rate)) if drop_rate else get_up_layer(ngf*4, ngf*4, mode=upmode, norm_layer=norm_layer, use_bias=use_bias)
        self.c8 = nn.Sequential(
                    nn.Conv3d(ngf*4 + ngf*2, ngf*2,3, padding=1),
                    norm_layer(ngf*2),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf*2, ngf*2,3, padding=1),
                    norm_layer(ngf*2),
                    nn.ReLU(inplace=True)
                    )

        self.u9 = nn.Sequential(get_up_layer(ngf*2, ngf*2, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), nn.Dropout(drop_rate)) if drop_rate else get_up_layer(ngf*2, ngf*2, mode=upmode, norm_layer=norm_layer, use_bias=use_bias)
        self.c10 = nn.Sequential(
                    nn.Conv3d(ngf*2 + ngf, ngf,3, padding=1),
                    norm_layer(ngf),
                    nn.ReLU(inplace=True),
                    nn.Conv3d(ngf, ngf,3, padding=1),
                    norm_layer(ngf),
                    nn.ReLU(inplace=True)
                    )

        self.out = nn.Sequential(nn.Conv3d(ngf, output_nc, 1 ), nn.Sigmoid() if use_sigmoid else nn.Tanh())
    def forward(self, x):
        x1 = self.c1(x)
        x2 = self.c2(self.d1(x1))
        x3 = self.c3(self.d2(x2)) 
        x10 = self.c10(torch.cat((self.u9(self.c8(torch.cat((self.u7(self.c6(torch.cat((self.u5(self.c4(self.d3(x3))),x3),1))),x2),1))),x1),1))
        return self.out(x10)

##############################################################################
# Small Unet 3D
##############################################################################

class UnetGenerator3D(nn.Module):
    """Create a Unet-based generator"""

    def __init__(self, input_nc=1, output_nc=1, num_downs=4, ngf=64, scale_factor=0, upmode='pixelshuffle', norm_layer=nn.BatchNorm3d, drop_rate=0, use_sigmoid=False):
        """Construct a Unet generator
        Parameters:
            input_nc (int):         -- the number of channels in input images
            output_nc (int):        -- the number of channels in output images
            num_downs (int):        -- the number of downsamplings in UNet. For example, # if |num_downs| == 7, image of size 128x128x128 will become of size 1x1x1 # at the bottleneck
            ngf (int):              -- the number of filters in the last conv layer
            scale_factor (int):     -- how much should the output be scaled up compared to the input
            upmode (str):           -- upsampling mode transposedconv | interpolation | pixelshuffle
            norm_layer (object):    -- normalization layer
            drop_rate (float):      -- use dropout layers if drop_rate > 0
            use_sigmoid (bool):     -- use sigmoid or tanh as last activation funnction

        We construct the U-Net from the innermost layer to the outermost layer.
        It is a recursive process.
        """
        super(UnetGenerator3D, self).__init__()
        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d
        # construct unet structure
        num_downs = max(num_downs, 3)
        for i in range(num_downs - 1): # add intermediate layers with maximal ngf * 8 filters
            filter_factor = min(8, 2**(num_downs-i-2))
            if i == 0:
                unet_block = UnetSkipConnectionBlock(ngf * filter_factor, ngf * filter_factor, upmode=upmode, input_nc=None, submodule=None, norm_layer=norm_layer, innermost=True)
            else:
                # gradually reduce the number of filters from ngf * factor to ngf
                unet_block = UnetSkipConnectionBlock(ngf * filter_factor, ngf * filter_factor_prev, upmode=upmode, input_nc=None, submodule=unet_block, norm_layer=norm_layer, drop_rate=drop_rate)
            filter_factor_prev = filter_factor 
        self.model = UnetSkipConnectionBlock(output_nc, ngf, upmode=upmode, input_nc=input_nc, submodule=unet_block, outermost=True, scale_factor=scale_factor, norm_layer=norm_layer, use_sigmoid=use_sigmoid)  # add the outermost layer

    def forward(self, input):
        """Standard forward"""
        return self.model(input)


class UnetSkipConnectionBlock(nn.Module):
    """Defines the Unet submodule with skip connection.
        X -------------------identity----------------------
        |-- downsampling -- |submodule| -- upsampling --|
    """

    def __init__(self, outer_nc, inner_nc, input_nc=None, submodule=None, outermost=False, innermost=False, scale_factor=0, upmode='pixelshuffle', norm_layer=nn.BatchNorm3d, drop_rate=0, use_sigmoid=False):
        """Construct a Unet submodule with skip connections.

        Parameters:
            outer_nc (int):                         -- the number of filters in the outer conv layer
            inner_nc (int):                         -- the number of filters in the inner conv layer
            input_nc (int):                         -- the number of channels in input images/features
            submodule (UnetSkipConnectionBlock):    -- previously defined submodules
            outermost (bool):                       -- if this module is the outermost module
            innermost (bool):                       -- if this module is the innermost module
            scale_factor (int):                     -- how much should the output be scaled up compared to the input
            upmode (str):                           -- upsampling mode transposedconv | interpolation | pixelshuffle
            norm_layer (object):                    -- normalization layer
            drop_rate (float):                      -- use dropout layers if drop_rate > 0
            use_sigmoid (bool):                     -- use sigmoid or tanh as last activation funnction
        """
        super(UnetSkipConnectionBlock, self).__init__()
        self.outermost = outermost
        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d
        if input_nc is None:
            input_nc = outer_nc
        downconv = nn.Conv3d(input_nc, inner_nc, kernel_size=4,
                             stride=2, padding=1, bias=use_bias)
        downrelu = nn.LeakyReLU(0.2, True)
        downnorm = norm_layer(inner_nc)
        uprelu = nn.ReLU(True)
        upnorm = norm_layer(outer_nc)
        if outermost:

            if scale_factor == 2:
                upconv = nn.Sequential(get_up_layer(inner_nc * 2, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), get_up_layer(inner_nc, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=False, trans_kernel=4, trans_pad=1))  # add the outermost layer with 2x upscaling
            elif scale_factor == 4:
                upconv = nn.Sequential(get_up_layer(inner_nc * 2, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), get_up_layer(inner_nc, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias), get_up_layer(inner_nc, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=False, trans_kernel=4, trans_pad=1))  # add the outermost layer with 4x upscaling
            else:
                upconv = get_up_layer(inner_nc * 2, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=False, trans_kernel=4, trans_pad=1)

            down = [downconv, downnorm]
            last = nn.Sigmoid() if use_sigmoid else nn.Tanh()
            #lastconv = [nn.Conv3d(inner_nc, inner_nc, kernel_size=3, stride=1, padding=1, bias=use_bias), nn.ReLU(True), norm_layer(inner_nc) ,nn.Conv3d(inner_nc, outer_nc, kernel_size=3, stride=1, padding=1, bias=use_bias)] # additional last convolutional layer
            lastconv = [nn.Conv3d(inner_nc, outer_nc, kernel_size=3, stride=1, padding=1, bias=use_bias)]
            up = [uprelu, upconv] +  lastconv + [last]
            model = down + [submodule] + up
        elif innermost:
            upconv = get_up_layer(inner_nc, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias)
            subconv = nn.Conv3d(inner_nc, outer_nc, kernel_size=3, stride=1, padding=1, bias=use_bias)
            up = [uprelu, upconv, subconv, upnorm]
            #up = [downrelu, upconv, subconv, upnorm]                         
            down = [downrelu, downconv, downnorm]
            if drop_rate > 0:
                model = down + up + [nn.Dropout(drop_rate)]
            else:
                model = down + up
        else:
            upconv = get_up_layer(inner_nc * 2, inner_nc, mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=False, trans_kernel=4, trans_pad=1)
            subconv = nn.Conv3d(inner_nc, outer_nc, kernel_size=3, stride=1, padding=1, bias=use_bias)
            down = [downrelu, downconv, downnorm]
            up = [uprelu, upconv, subconv, upnorm]

            if drop_rate > 0:
                model = down + [submodule] + up + [nn.Dropout(drop_rate)]
            else:
                model = down + [submodule] + up

        self.model = nn.Sequential(*model)

    def forward(self, x):
        if self.outermost:
            return self.model(x)
        else:   # add skip connections
            return torch.cat([x, self.model(x)], 1)

##############################################################################
# ResNet 3D
##############################################################################

class ResnetGenerator3D(nn.Module):
    """Resnet-based generator that consists of Resnet blocks between a few downsampling/upsampling operations.

    We adapt Torch code and idea from Justin Johnson's neural style transfer project(https://github.com/jcjohnson/fast-neural-style)
    """

    def __init__(self, input_nc, output_nc, ngf=32, n_blocks=6, padding_type='replicate', upmode='transposedconv', norm_layer=nn.BatchNorm3d, drop_rate=0, use_sigmoid=False):
        """Construct a Resnet-based generator

        Parameters:
            input_nc (int):         -- the number of channels in input images
            output_nc (int):        -- the number of channels in output images
            ngf (int):              -- the number of filters in the last conv layer
            n_blocks (int):         -- the number of ResNet blocks
            padding_type (str):     -- the name of padding layer in conv layers: replicate | zero
            upmode (str):           -- upsampling mode transposedconv | interpolation | pixelshuffle
            norm_layer (object):    -- normalization layer
            drop_rate (float):      -- use dropout layers if drop_rate > 0
            use_sigmoid (bool):     -- use sigmoid or tanh as last activation funnction
        """
        assert(n_blocks >= 0)
        super(ResnetGenerator3D, self).__init__()
        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        model = [nn.ReplicationPad3d(3),
                 nn.Conv3d(input_nc, ngf, kernel_size=7, padding=0, bias=use_bias),
                 norm_layer(ngf),
                 nn.ReLU(True)]

        n_downsampling = 2
        for i in range(n_downsampling):  # add downsampling layers
            mult = 2 ** i
            model += [nn.Conv3d(ngf * mult, ngf * mult * 2, kernel_size=3, stride=2, padding=1, bias=use_bias),
                      norm_layer(ngf * mult * 2),
                      nn.ReLU(True)]

        mult = 2 ** n_downsampling
        for i in range(n_blocks):       # add ResNet blocks

            model += [ResnetBlock3D(ngf * mult, padding_type=padding_type, norm_layer=norm_layer, drop_rate=drop_rate, use_bias=use_bias)]

        for i in range(n_downsampling):  # add upsampling layers
            mult = 2 ** (n_downsampling - i)
            model += [get_up_layer(ngf * mult, int(ngf * mult / 2), mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=False, trans_kernel=3, trans_pad=1, trans_output_padding=1)]
        model += [nn.ReplicationPad3d(3)]
        model += [nn.Conv3d(ngf, output_nc, kernel_size=7, padding=0)]
        model += [nn.Tanh() if not use_sigmoid else nn.Sigmoid()]

        self.model = nn.Sequential(*model)
    def forward(self, input):
        """Standard forward"""
        return self.model(input)

class ResnetBlock3D(nn.Module):
    """Define a Resnet block"""

    def __init__(self, dim, padding_type, norm_layer, drop_rate, use_bias):
        """Initialize the Resnet block

        A resnet block is a conv block with skip connections
        We construct a conv block with build_conv_block function,
        and implement skip connections in <forward> function.
        Original Resnet paper: https://arxiv.org/pdf/1512.03385.pdf
        """
        super(ResnetBlock3D, self).__init__()
        self.conv_block = self.build_conv_block(dim, padding_type, norm_layer, drop_rate, use_bias)

    def build_conv_block(self, dim, padding_type, norm_layer, drop_rate, use_bias):
        """Construct a convolutional block.

        Parameters:
            dim (int):              -- the number of channels in the conv layer.
            padding_type (str):     -- the name of padding layer: replicate | zero
            norm_layer (object):    -- normalization layer
            drop_rate (float):      -- use dropout layers if drop_rate > 0.
            use_bias (bool):        -- if the conv layer uses bias or not

        Returns a conv block (with a conv layer, a normalization layer, and a non-linearity layer (ReLU))
        """
        conv_block = []
        p = 0
        if padding_type == 'replicate':
            conv_block += [nn.ReplicationPad3d(1)]
        elif padding_type == 'zero':
            p = 1
        else:
            raise NotImplementedError('padding [%s] is not implemented' % padding_type)

        conv_block += [nn.Conv3d(dim, dim, kernel_size=3, padding=p, bias=use_bias), norm_layer(dim), nn.ReLU(True)]
        if drop_rate:
            conv_block += [nn.Dropout(drop_rate)]

        p = 0
        if padding_type == 'replicate':
            conv_block += [nn.ReplicationPad3d(1)]
        elif padding_type == 'zero':
            p = 1
        else:
            raise NotImplementedError('padding [%s] is not implemented' % padding_type)
        conv_block += [nn.Conv3d(dim, dim, kernel_size=3, padding=p, bias=use_bias), norm_layer(dim), nn.ReLU(True)]

        return nn.Sequential(*conv_block)

    def forward(self, x):
        """Forward function (with skip connections)"""
        out = x + self.conv_block(x)  # add skip connections
        return out

##############################################################################
# DenseNet 3D
##############################################################################

class DenseNet3D(nn.Module):
    """Densenet-BC model class, based on
    Densely Connected Convolutional Networks" <https://arxiv.org/pdf/1608.06993.pdf>
    Args:
        input_nc (int):                     -- number of input channels
        growth_rate (int):                  -- how many filters to add each layer (`k` in paper)
        block_config (list of 3 or 4 ints): -- how many layers in each pooling block
        num_init_features (int):            -- the number of filters to learn in the first convolution layer
        bn_size (int):                      -- multiplicative factor for number of bottle neck layers (i.e. bn_size * k features in the bottleneck layer)
        small_inputs (bool):                -- set to True if images are 32x32. Otherwise assumes images are larger.
        efficient (bool):                   -- set to True to use checkpointing. Much more memory efficient, but slower.
        upmode (str):                       -- upsampling mode transposedconv | interpolation | pixelshuffle
        norm_layer (object):                -- normalization layer
        drop_rate (float):                  -- use dropout layers if drop_rate > 0
        use_sigmoid (bool):                 -- use sigmoid or tanh as last activation funnction
    """
    def __init__(self, input_nc=1, output_nc=1, growth_rate=16, block_config=(16, ), compression=0.5, num_init_features=32, bn_size=4, small_inputs=True, efficient=False, upmode='transposedconv', norm_layer=nn.BatchNorm3d, drop_rate=0, use_sigmoid=True):

        super(DenseNet3D, self).__init__()
        assert 0 < compression <= 1, 'compression of densenet should be between 0 and 1'
        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        self.avgpool_size = 7

        # First convolution
        self.features = nn.Sequential(OrderedDict([('conv0', nn.Conv3d(input_nc, num_init_features, kernel_size=7, stride=2, padding=3, bias=use_bias)),]))
        self.features.add_module('norm0', norm_layer(num_init_features))
        self.features.add_module('relu0', nn.ReLU(inplace=True))
        self.features.add_module('pool0', nn.MaxPool3d(kernel_size=3, stride=2, padding=1,
                                                           ceil_mode=False))

        # Each denseblock
        num_features = num_init_features
        for i, num_layers in enumerate(block_config):
            block = _DenseBlock(
                num_layers=num_layers,
                num_input_features=num_features,
                bn_size=bn_size,
                growth_rate=growth_rate,
                drop_rate=drop_rate,
                efficient=efficient,
                norm_layer=norm_layer,
                use_bias=use_bias
            )
            self.features.add_module('denseblock%d' % (i + 1), block)
            num_features = num_features + num_layers * growth_rate
            if i != len(block_config) - 1:
                trans = _Transition(num_input_features=num_features,
                                    num_output_features=int(num_features * compression), norm_layer=norm_layer, use_bias=use_bias)
                self.features.add_module('transition%d' % (i + 1), trans)
                num_features = int(num_features * compression)

        # Final batch norm
        self.features.add_module('norm_final', norm_layer(num_features))
        self.features.add_module('relu_final', nn.ReLU(True))
        for i, num_layers in enumerate(block_config):
            mult = 2 ** i
            self.features.add_module('upsampling%d' % (i + 1), get_up_layer( num_features//mult, num_features//(mult*2), mode=upmode, use_bias=use_bias, norm_layer=norm_layer, last=False, trans_kernel=4, trans_pad=1, trans_output_padding=0))
        self.features.add_module('last_upsampling', get_up_layer( num_features//(mult*2), num_features//(mult*4), mode=upmode, use_bias=use_bias, norm_layer=norm_layer, last=False, trans_kernel=4, trans_pad=1, trans_output_padding=0))
        self.features.add_module('pad', nn.ReplicationPad3d(3))
        self.features.add_module('lastconv', nn.Conv3d(num_features//(mult*4), output_nc, kernel_size=7, padding=0))
        self.features.add_module('sigmoid', nn.Sigmoid() if use_sigmoid else nn.Tanh())

    def forward(self, x):
        features = self.features(x)
        return features

class _DenseBlock(nn.Module):
    def __init__(self, num_layers, num_input_features, bn_size, growth_rate, drop_rate, efficient=False, norm_layer=nn.BatchNorm3d, use_bias=False):
        super(_DenseBlock, self).__init__()
        for i in range(num_layers):
            layer = _DenseLayer(
                num_input_features + i * growth_rate,
                growth_rate=growth_rate,
                bn_size=bn_size,
                drop_rate=drop_rate,
                efficient=efficient,
                norm_layer=norm_layer,
                use_bias=use_bias
            )
            self.add_module('denselayer%d' % (i + 1), layer)

    def forward(self, init_features):
        features = [init_features]
        for name, layer in self.named_children():
            new_features = layer(*features)
            features.append(new_features)
        return torch.cat(features, 1)

class _Transition(nn.Sequential):
    def __init__(self, num_input_features, num_output_features, norm_layer=nn.BatchNorm3d, use_bias=False):
        super(_Transition, self).__init__()
        self.add_module('norm', norm_layer(num_input_features))
        self.add_module('relu', nn.ReLU(inplace=True))
        self.add_module('conv', nn.Conv3d(num_input_features, num_output_features,
                                          kernel_size=1, stride=1, bias=use_bias))
        self.add_module('pool', nn.AvgPool3d(kernel_size=2, stride=2))

class _DenseLayer(nn.Module):
    def __init__(self, num_input_features, growth_rate, bn_size, drop_rate, efficient=False, norm_layer=nn.BatchNorm3d, use_bias=False):
        super(_DenseLayer, self).__init__()
        self.add_module('norm1', norm_layer(num_input_features)),
        self.add_module('relu1', nn.ReLU(inplace=True)),
        self.add_module('conv1', nn.Conv3d(num_input_features, bn_size * growth_rate,
                        kernel_size=1, stride=1, bias=use_bias)),
        self.add_module('norm2', norm_layer(bn_size * growth_rate)),
        self.add_module('relu2', nn.ReLU(inplace=True)),
        self.add_module('conv2', nn.Conv3d(bn_size * growth_rate, growth_rate,
                        kernel_size=3, stride=1, padding=1, bias=use_bias)),
        self.drop_rate = drop_rate
        self.efficient = efficient

    def forward(self, *prev_features):
        bn_function = _bn_function_factory(self.norm1, self.relu1, self.conv1)
        if self.efficient and any(prev_feature.requires_grad for prev_feature in prev_features):
            bottleneck_output = cp.checkpoint(bn_function, *prev_features)
        else:
            bottleneck_output = bn_function(*prev_features)
        new_features = self.conv2(self.relu2(self.norm2(bottleneck_output)))
        if self.drop_rate > 0:
            new_features = F.dropout(new_features, p=self.drop_rate, training=self.training)
        return new_features

##############################################################################
# RRDBNet 3D
##############################################################################

class ResidualDenseBlock_5C(nn.Module):
    def __init__(self, nf=64, gc=32, use_bias=True, norm_layer=None):
        super(ResidualDenseBlock_5C, self).__init__()
        # gc: growth channel, i.e. intermediate channels
        self.conv1 = nn.Conv3d(nf, gc, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf, gc, 3, 1, 1, bias=use_bias), norm_layer(gc))
        self.conv2 = nn.Conv3d(nf + gc, gc, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf + gc, gc, 3, 1, 1, bias=use_bias), norm_layer(gc))
        self.conv3 = nn.Conv3d(nf + 2 * gc, gc, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf + 2 * gc, gc, 3, 1, 1, bias=use_bias), norm_layer(gc))
        self.conv4 = nn.Conv3d(nf + 3 * gc, gc, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf + 3 * gc, gc, 3, 1, 1, bias=use_bias), norm_layer(gc))
        self.conv5 = nn.Conv3d(nf + 4 * gc, nf, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf + 4 * gc, nf, 3, 1, 1, bias=use_bias), norm_layer(nf))
        self.lrelu = nn.LeakyReLU(negative_slope=0.2, inplace=True)

    def forward(self, x):
        x1 = self.lrelu(self.conv1(x))
        x2 = self.lrelu(self.conv2(torch.cat((x, x1), 1)))
        x3 = self.lrelu(self.conv3(torch.cat((x, x1, x2), 1)))
        x4 = self.lrelu(self.conv4(torch.cat((x, x1, x2, x3), 1)))
        x5 = self.conv5(torch.cat((x, x1, x2, x3, x4), 1))
        return x5 * 0.2 + x


class RRDB(nn.Module):
    '''Residual in Residual Dense Block'''

    def __init__(self, nf, gc=32, use_bias=True, norm_layer=None):
        super(RRDB, self).__init__()
        self.RDB1 = ResidualDenseBlock_5C(nf, gc, norm_layer=norm_layer, use_bias=use_bias)
        self.RDB2 = ResidualDenseBlock_5C(nf, gc, norm_layer=norm_layer, use_bias=use_bias)
        self.RDB3 = ResidualDenseBlock_5C(nf, gc, norm_layer=norm_layer, use_bias=use_bias)

    def forward(self, x):
        out = self.RDB1(x)
        out = self.RDB2(out)
        out = self.RDB3(out)
        return out * 0.2 + x


class RRDBNet3D(nn.Module):
    """
    Residual in Residual Dense Net from https://arxiv.org/abs/1809.00219
    """
    def __init__(self, in_nc, out_nc, nf, nb, gc=32, num_downs=2, upmode='interpolation', norm_layer=None, use_sigmoid=False):
        super(RRDBNet3D, self).__init__()

        if type(norm_layer) == functools.partial:
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        RRDB_block_f = functools.partial(RRDB, nf=nf, gc=gc, norm_layer=norm_layer, use_bias=use_bias)

        downs = []

        num_downs = 2
        num_in = in_nc
        ngf = nf//(2**num_downs)
        for i in range(num_downs):  # add downsampling layers
            mult = 2 ** i
            downs += [nn.Conv3d(num_in, ngf * mult, kernel_size=3, stride=2, padding=1, bias=use_bias)]
            if not norm_layer is None:
                downs += [norm_layer(ngf * mult)]
            downs += [nn.LeakyReLU(negative_slope=0.2, inplace=True)]
            num_in = ngf * mult

        self.downs = nn.Sequential(*downs)
        self.conv_first = nn.Conv3d(num_in, nf, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(num_in, nf, 3, 1, 1, bias=use_bias), norm_layer(nf))

        layers = []
        for _ in range(nb):
            layers.append(RRDB_block_f())        
        self.RRDB_trunk = nn.Sequential(*layers)

        self.trunk_conv = nn.Conv3d(nf, nf, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf, nf, 3, 1, 1, bias=use_bias), norm_layer(nf))
        #### upsampling

        ups = []
        for i in range(num_downs):  # add upsampling layers
            ups += [get_up_layer(nf, nf, mode=upmode, norm_layer=norm_layer, use_bias=use_bias, last=True, trans_kernel=4, trans_pad=1)]
            if not norm_layer is None:
                ups += [norm_layer(ngf * mult)]
            ups += [nn.LeakyReLU(negative_slope=0.2, inplace=use_bias)]
        self.ups = nn.Sequential(*ups)

        self.HRconv = nn.Conv3d(nf, nf, 3, 1, 1, bias=use_bias) if norm_layer is None else nn.Sequential(nn.Conv3d(nf, nf, 3, 1, 1, bias=use_bias), norm_layer(nf))
        self.conv_last = nn.Conv3d(nf, out_nc, 3, 1, 1, bias=use_bias)
        if use_sigmoid:
            self.conv_last = nn.Sequential(self.conv_last, nn.Sigmoid())
        self.lrelu = nn.LeakyReLU(negative_slope=0.2, inplace=use_bias)

    def forward(self, x):
        down = self.downs(x)
        fea = self.conv_first(down)
        trunk = self.trunk_conv(self.RRDB_trunk(fea))
        fea = fea + trunk

        fea = self.ups(fea)
        out = self.conv_last(self.lrelu(self.HRconv(fea)))

        return out

##############################################################################
# Discriminator Network Architectures
##############################################################################

class NLayerDiscriminator3D(nn.Module):
    """Defines a PatchGAN discriminator for volumetric data"""

    def __init__(self, input_nc=1, ndf=64, n_layers=3, norm_layer=nn.BatchNorm3d):
        """Construct a PatchGAN discriminator

        Parameters:
            input_nc (int):         -- the number of channels in input images
            ndf (int):              -- the number of filters in the last conv layer
            n_layers (int):         -- the number of conv layers in the discriminator
            norm_layer (object):    -- normalization layer
        """
        super(NLayerDiscriminator3D, self).__init__()
        if type(norm_layer) == functools.partial:  # no need to use bias as BatchNorm3d has affine parameters
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        kw = 4
        padw = 1
        sequence = [nn.Conv3d(input_nc, ndf, kernel_size=kw, stride=2, padding=padw), norm_layer(ndf), nn.LeakyReLU(0.2, True)]
        nf_mult = 1
        nf_mult_prev = 1
        for n in range(1, n_layers):  # gradually increase the number of filters
            nf_mult_prev = nf_mult
            nf_mult = min(2 ** n, 8)
            sequence += [
                nn.Conv3d(ndf * nf_mult_prev, ndf * nf_mult, kernel_size=kw, stride=2, padding=padw, bias=use_bias),
                norm_layer(ndf * nf_mult),
                nn.LeakyReLU(0.2, True)
            ]

        sequence += [nn.Conv3d(ndf * nf_mult, 1, kernel_size=kw, stride=2, padding=padw)]  # output 1 channel prediction map
        self.model = nn.Sequential(*sequence)

    def forward(self, input):
        """Standard forward."""
        return self.model(input)

class PixelDiscriminator3D(nn.Module):
    """Defines a 1x1 PatchGAN discriminator (pixelGAN)"""

    def __init__(self, input_nc, ndf=64, norm_layer=nn.BatchNorm3d):
        """Construct a 1x1 PatchGAN discriminator

        Parameters:
            input_nc (int):       -- the number of channels in input images
            ndf (int):            -- the number of filters in the last conv layer
            norm_layer (object):  -- normalization layer
        """
        super(PixelDiscriminator3D, self).__init__()
        if type(norm_layer) == functools.partial:  # no need to use bias as BatchNorm3d has affine parameters
            use_bias = norm_layer.func != nn.BatchNorm3d
        else:
            use_bias = norm_layer != nn.BatchNorm3d

        self.net = [
            nn.Conv3d(input_nc, ndf, kernel_size=1, stride=1, padding=0),
            nn.LeakyReLU(0.2, True),
            nn.Conv3d(ndf, ndf * 2, kernel_size=1, stride=1, padding=0, bias=use_bias),
            norm_layer(ndf * 2),
            nn.LeakyReLU(0.2, True),
            nn.Conv3d(ndf * 2, 1, kernel_size=1, stride=1, padding=0, bias=use_bias)]

        self.net = nn.Sequential(*self.net)

    def forward(self, input):
        """Standard forward."""
        return self.net(input)

def _bn_function_factory(norm, relu, conv):
    def bn_function(*inputs):
        concated_features = torch.cat(inputs, 1)
        bottleneck_output = conv(relu(norm(concated_features)))
        return bottleneck_output

    return bn_function

##############################################################################
# General Blocks
##############################################################################

def get_up_layer( inner_nc, outer_nc, mode='pixelshuffle', norm_layer=nn.BatchNorm3d, use_bias=False, last=False, trans_kernel=2, trans_pad=0, trans_output_padding=0):
        if mode == 'pixelshuffle':
            upconv =  PixelShuffleResize3dBlock(2, inner_nc, outer_nc, 3, norm_layer=norm_layer, padding=1, last=last, use_bias=use_bias)
        elif mode == 'interpolation':
            if last:
                upconv = ResizeConv3dBlock(2, inner_nc, outer_nc, 3, padding=1)
            else:
                upconv = nn.Sequential(ResizeConv3dBlock(2, inner_nc, outer_nc, 3, padding=1), norm_layer(outer_nc), nn.ReLU(True))
        elif mode == 'transposedconv':
            if last:
                upconv = nn.ConvTranspose3d(inner_nc, outer_nc, kernel_size=trans_kernel, stride=2, padding=trans_pad, output_padding=trans_output_padding, bias=use_bias)
            else:
                upconv = nn.Sequential(nn.ConvTranspose3d(inner_nc, outer_nc, kernel_size=trans_kernel, stride=2, padding=trans_pad, output_padding=trans_output_padding, bias=use_bias), norm_layer(outer_nc), nn.ReLU(True))
        return upconv

class ResizeConv3dBlock(nn.Module):
    """
        First resizes the block with 'nearest' or 'bilinear' interpolation and then adds a convolutional block 
    """
    def __init__(self, scale_factor, in_channels, out_channels, kernel_size, mode='nearest', stride=1, padding=0, dilation=1, groups=1, use_bias=True):
        super(ResizeConv3dBlock, self).__init__()
        self.mode = mode
        self.scale_factor = scale_factor
        self.conv = nn.Conv3d(in_channels, out_channels, kernel_size, stride=stride, padding=padding, dilation=dilation, groups=groups, bias=use_bias)
    def forward(self,x):
        return self.conv(nn.functional.interpolate(x, scale_factor=self.scale_factor, mode=self.mode))

class PixelShuffleResize3dBlock(nn.Module):
    """
        Pixel shuffle layer extended to support volumetric data
        (Real-Time Single Image and Video Super-Resolution Using an Efficient Sub-Pixel Convolutional Neural Network, CVPR17)
    """
    def __init__(self, scale_factor, in_channels, out_channels, kernel_size, stride=1, padding=1, dilation=1, groups=1, norm_layer=nn.BatchNorm3d, last=False, use_bias=True):
        super(PixelShuffleResize3dBlock, self).__init__()
        self.scale_factor = scale_factor
        if last:
            self.upconv = nn.Conv3d(in_channels, out_channels*(scale_factor**3), 3, stride=1, padding=1, dilation=1, bias=use_bias)
        else:
            self.upconv = nn.Sequential(nn.Conv3d(in_channels, out_channels*(scale_factor**3), 3, stride=1, padding=1, dilation=1, bias=use_bias), norm_layer(out_channels*(scale_factor**3)), nn.ReLU() )
        self.up = PixelShuffle3d(scale_factor)
    def forward(self,x):
        return self.up(self.upconv(x))

class Identity(nn.Module):
    """
        Simple Class to add no normalization layer
    """
    def __init__(self, *args, **kwargs):
        super().__init__()
    def forward(self, x):
        return x
