import random
from agents.api_repo import *
from typing import Dict, Any
import json
import pandas as pd
import os


class MockAPIHandler:
    def __init__(self):
        # Defining error types
        self.fail_calling_type = {404: 'API not found',
                     400: 'For APIs without parameters, errors are generated with parameters.',
                     500: 'The API parameter format is invalid.',
                     501: 'Unable to retrieve the API parameter rules.',
                     502: 'API parameter rule format error',
                     422: 'API action_code generates non-existent parameter names.',
                     403: 'API action_code is missing required parameters.',
                     406: 'The API action_code contains parameters with empty content.',
                     409: 'The API action_code parameter does not conform to the enumerated value type.',
                     300: 'No API return result was found.'}

        print(os.getcwd())

        api_obs_map_df = pd.read_csv('../agent/api_obs_mock_data/toolbench_obs_map_0515.csv')
        self.toolbench_obs_map_lines = {}
        for idx, row in api_obs_map_df.iterrows():
            self.toolbench_obs_map_lines[row['tool']] = row['values']
        self.toolbench_api = list(self.toolbench_obs_map_lines.keys())

        with open("../agent/api_obs_mock_data/check_shop_expenses_api_mock_query_response_0515.json") as f:
            lines = f.readlines()
        self.check_shop_expenses_api_res = eval(lines[0])

    def sort_and_match_dict(self, input_dict: Dict[str, Any], sorted_params_flag=True, api_params_key='parameter') -> str:
        """
        对输入的字典进行排序
        """
        # 对字典的键值对进行排序
        try:
            sorted_dict = {k: sorted(v) if isinstance(v, list) else v for k, v in sorted(input_dict.items())}
            if sorted_params_flag:
                if api_params_key in sorted_dict and isinstance(sorted_dict[api_params_key], dict):
                    sorted_dict[api_params_key] = {k: sorted(v) if isinstance(v, list) else v for k, v in
                                                sorted(sorted_dict[api_params_key].items())}
        except:
            sorted_dict = input_dict

        return sorted_dict

    def call_api(self, query, api_calling_params, response_flag=True):
        """
        Simulating API calls
        :param api_calling_params: API name + parameter dictionary passed in. eg.{"api_name": "check_shop_qualifications", "parameter": {"keyword": ["vegetables"]}}
        :return: Return Response
        """
        response = ''
        if not isinstance(api_calling_params, dict):
            return self.fail_calling_type[500]
        api_name = api_calling_params.get('api_name', '')
        params = api_calling_params.get('parameter', {})

        # 1.Check if the API exists
        if api_name not in RODY_TOOL_LIST:
            return self.fail_calling_type[404]
        api_rule_config = RODY_TOOL_DICT[api_name]
        api_config = api_rule_config.get('parameters', None)

        # 2.Perform parameter verification
        valid_status_code = self._validate_params(params, api_config)
        if valid_status_code != 200:
            response_flag = False
            response = self.fail_calling_type[valid_status_code]

        # 3.Simulate parameter calls
        # 3.1 Verify the mock situation
        sorted_api_calling_params = self.sort_and_match_dict(api_calling_params)
        sorted_api_calling_params = str(sorted_api_calling_params)
        if not response:
            responses_config = api_rule_config.get('mock_response', {})
            if not isinstance(responses_config, dict):
                response_flag = False
                response = self.fail_calling_type[300]
            else:
                response = responses_config.get('success_responses', "") if response_flag else responses_config.get('error_responses', "")
                if not response:
                    response = self.fail_calling_type[300]

                # 3.2 Generate mock results
                if response_flag and api_name == 'check_shop_expenses':
                    shop_expenses_api_res = self.check_shop_expenses_api_res.get(query, None)
                    if shop_expenses_api_res is None:
                        response = self.fail_calling_type[300]
                    else:
                        mock_fee, transaction_rate, technical_rate = shop_expenses_api_res
                        response = f"The relevant operating fees are {str(mock_fee)} yuan."
                        response = f"1.Specific amount of margin per year: {str(mock_fee)} yuan.\n2.Transaction service fee rate per transaction: {str(transaction_rate)}%.\n3.Technical service fee rate per transaction:{str(technical_rate)}%."

        prefix = f'Pass calling {sorted_api_calling_params}, the API call was successful.\nResults are as follows: ' if response_flag else \
            f'Pass calling {sorted_api_calling_params}, the API call was failed.\nResults are as follows: '

        response = prefix + response

        return response

    def call_api_common(self, api_calling_params, api_name_key='', api_params_key=''):
        """
        Simulate toolbench-API calls. Due to the lack of API rules, no complex verification is performed. Only a dictionary format is verified, and then the previously defined API+action corresponding return is directly read as the response.
        :param api_calling_params: API name + parameter dictionary passed in
        :return: Return Response
        """
        def action_input_update(x):
            """
            Convert the action parameter to a dictionary for easy comparison
            """
            if isinstance(x, dict):
                return x
            elif isinstance(x, str):
                try:
                    new_x = x.replace('true', 'True').replace('false', 'False')
                    return eval(new_x)
                except:
                    return x
            else:
                return None

        if not isinstance(api_calling_params, dict):
            return self.fail_calling_type[500]

        ## Compatible with input data formats
        if not api_name_key or not api_params_key:
            if 'api_name' in api_calling_params.keys() and 'parameter' in api_calling_params.keys():
                api_name_key = 'api_name'
                api_params_key = 'parameter'
            elif 'tool' in api_calling_params.keys() and 'action_input' in api_calling_params.keys():
                api_name_key = 'tool'
                api_params_key = 'action_input'
            else:
                return self.fail_calling_type[500]

        ## 1.Parameter reading
        api_calling_params[api_params_key] = action_input_update(api_calling_params[api_params_key])  ## Convert the parameters into a dictionary
        sorted_api_calling_params = self.sort_and_match_dict(api_calling_params, api_params_key=api_params_key)  ## Sort the overall incoming parameters

        api_name = sorted_api_calling_params[api_name_key]
        api_params = sorted_api_calling_params[api_params_key]
        for parameter_name, parameter_value_list in api_params.items():
            # Make sure the data structure of parameter_value_list is ','.join(list)
            api_params[parameter_name] = ','.join([str(v) for v in parameter_value_list]) if isinstance(parameter_value_list, list) else str(parameter_value_list)
        api_params = str(api_params)

        ## 2.get response
        response = self.toolbench_obs_map_lines.get(api_name)
        if response is None:
            if api_name not in self.toolbench_api:
                return self.fail_calling_type[300]
            else:
                response = f'Pass calling {sorted_api_calling_params}, the API call was successful, but API return result was not found.'
                return response

        sorted_api_calling_params = str(sorted_api_calling_params)
        response = f'Pass calling {sorted_api_calling_params}, the API call was successful.\nResults are as follows: ' + str(response)

        return response

    def _validate_params(self, parameter_dict, api_config):
        """
        Parameter Validation
        :param api_config: API Configuration
        :param parameter_dict: parameters
        :return: Status Code
        """
        # 1.Parameter rule verification
        ## 1.1 First check whether the rule exists
        if api_config is None or (not isinstance(api_config, dict)):
            return 501

        ## 1.2 Verification rule format
        api_param_config = api_config.get('properties', {})
        required_params = api_config.get('required', [])
        if (not isinstance(api_param_config, dict)) or (not isinstance(required_params, list)):
            return 502

        # 2.Verify the production of parameters without reference API:
        if len(api_param_config) == 0:
            return 200 if len(parameter_dict) == 0 else 400

        # 3.Parameter name validation
        valid_params_name = list(api_param_config.keys())
        all_params_name = list(parameter_dict.keys())
        if (set(all_params_name) - set(valid_params_name)) != set():
            return 422

        # 4.Required parameter verification
        if (set(required_params) - set(all_params_name)) != set():
            return 403

        # 5.Parameter corresponding value verification
        for parameter_name, parameter_value_list in parameter_dict.items():
            # 5.1 Make sure the data structure of parameter_value_list is list
            parameter_value_list = parameter_value_list if isinstance(parameter_value_list, list) else [
                parameter_value_list]
            parameter_dict[parameter_name] = parameter_value_list
            # 5.2 Check parameter_value
            for parameter_value in parameter_value_list:
                if not parameter_value:
                    return 406


        return 200



if __name__ == '__main__':
    handler = MockAPIHandler()
    action_code = {"api_name": "check_shop_qualifications", "parameter": {"keyword": ["apple"]}}
    obs = handler.call_api(action_code)
    """For APIs without parameters, errors are generated with parameters."""
    print(obs)
    action_code_common = {"tool": "check_status_for_background_removal_v2", "action_input": "{}"}
    obs = handler.call_api_common(action_code_common)
    print(obs)
    """Pass calling {'action_input': {}, 'tool': 'check_status_for_background_removal_v2'}, the API call was successful.
Results are as follows: {'status': True}"""