import pytorch_lightning as pl
import argparse
import pprint
from loguru import logger as loguru_logger

from src.config.PAConfig import get_cfg_defaults as gcd
from src.utils.profiler import build_profiler

from src.lightning.data import MultiSceneDataModule
from src.lightning.lightning_PAloftr import PL_PALoFTR
from src.lightning.lightning_loftr import PL_LoFTR

from pathlib import Path

data_cfg_path = "configs/data/scannet_100scene.py"
main_cfg_path = "configs/loftr/indoor/PAloftr_matcher.py"

# ------ checkpoint paths --------
ckpt_path = "logs/tb_logs/test-ds-bs=8/3dpe_baseline/checkpoints/PA-LoFTR_3dpe_only.ckpt"
dump_dir = "logs/tb_logs/test-ds-bs=8/3dpe_baseline/dump/"

is_loftr = False
if is_loftr:
    ckpt_path = "logs/tb_logs/indoor-ds-bs=8/loftr_full_baseline_30epo/checkpoints/epoch=21-auc@5=0.201-auc@10=0.369-auc@20=0.539.ckpt"
    dump_dir = "logs/tb_logs/indoor-ds-bs=8/loftr_full_baseline_30epo/dump/"
    main_cfg_path = "configs/loftr/indoor/loftr_ds_dense.py"
# --------------------------------

n_nodes = 1
n_gpus_per_node = 1
torch_num_workers = 4
batch_size = 1
pin_memory = True
exp_name = "test-ds-bs={}".format(n_gpus_per_node * n_nodes * batch_size)

MyArgs = [
    data_cfg_path,
    main_cfg_path,
    '--ckpt_path={}'.format(ckpt_path),
    '--dump_dir={}'.format(dump_dir),
    '--gpus={}'.format(n_gpus_per_node),
    '--num_nodes={}'.format(n_nodes),
    '--batch_size={}'.format(batch_size),
    '--num_workers={}'.format(torch_num_workers),
    '--num_sanity_val_steps=10',
    '--benchmark=True',
    '--max_epochs=10'
]


def parse_args(args=None):
    # init a costum parser which will be added into pl.Trainer parser
    # check documentation: https://pytorch-lightning.readthedocs.io/en/latest/common/trainer.html#trainer-flags
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(
        'data_cfg_path', type=str, help='data config path')
    parser.add_argument(
        'main_cfg_path', type=str, help='main config path')
    parser.add_argument(
        '--ckpt_path', type=str, default="weights/indoor_ds.ckpt", help='path to the checkpoint')
    parser.add_argument(
        '--dump_dir', type=str, default=None, help="if set, the matching results will be dump to dump_dir")
    parser.add_argument(
        '--profiler_name', type=str, default=None, help='options: [inference, pytorch], or leave it unset')
    parser.add_argument(
        '--batch_size', type=int, default=1, help='batch_size per gpu')
    parser.add_argument(
        '--num_workers', type=int, default=2)
    parser.add_argument(
        '--thr', type=float, default=None, help='modify the coarse-level matching threshold.')

    parser = pl.Trainer.add_argparse_args(parser)
    if args == None:
        return parser.parse_args()
    else:
        return parser.parse_args(args)


if __name__ == '__main__':
    # parse arguments
    args = parse_args(MyArgs)
    pprint.pprint(vars(args))
    
    # init default-cfg and merge it with the main- and data-cfg
    config = gcd()
    config.merge_from_file(args.main_cfg_path)
    config.merge_from_file(args.data_cfg_path)
    pl.seed_everything(config.TRAINER.SEED) # reproducibility
    
    # tune when testing
    if args.thr is not None:
        config.LOFTR.MATCH_COARSE.THR = args.thr
        
    loguru_logger.info(f"Args and config initialized!")
    
    # lightning module
    profiler = build_profiler(args.profiler_name)

    if is_loftr:
        model = PL_LoFTR(
            config,
            pretrained_ckpt=args.ckpt_path,
            profiler=profiler,
            dump_dir=args.dump_dir
        )
    else:
        model = PL_PALoFTR(
            config,
            pretrained_ckpt=args.ckpt_path,
            profiler=profiler,
            dump_dir=args.dump_dir
        )
    
    loguru_logger.info(f"Inst-Loftr lightning initialized!")
    
    # data
    data_module = MultiSceneDataModule(args, config)
    loguru_logger.info(f"DataModule initialized!")
    
    # trainer
    trainer = pl.Trainer.from_argparse_args(args, replace_sampler_ddp=False, logger=False)
    
    loguru_logger.info(f"Start testing!")
    trainer.test(model, datamodule=data_module, verbose=False)