import torch
import torch.nn as nn
from einops.einops import rearrange

from .backbone import build_backbone
from .utils.position_encoding import PositionEncodingSine
from .utils.position_encoding3d import PositionalEncoding3D
from .lotfr_module import LocalFeatureTransformer, FinePreprocess
from .depth_predictor.depth_predictor import DepthPredictor
from .depth_predictor.depth_transformer import DepthGuidedEncoder, TransformerEncoderGeneral
from .utils.coarse_matching import CoarseMatching
from .utils.fine_matching import FineMatching


class PALoFTR(nn.Module):
    def __init__(self, config):
        super().__init__()
        # Misc
        self.config = config
        
        # Modules
        self.backbone = build_backbone(config) # ResNet + FPN here
        # self.pos_encoding = PositionEncodingSine(
        #     config['coarse']['d_model'],
        #     temp_bug_fix=config['coarse']['temp_bug_fix']
        # )
        self.pos_encoding = PositionalEncoding3D(config['pe'])
        # self.depth_predictor = DepthPredictor(config['depth_predictor'])
        self.loftr_coarse = TransformerEncoderGeneral(config['depth_loftr'])
        self.coarse_matching = CoarseMatching(config['match_coarse'])
        self.fine_preprocess = FinePreprocess(config)
        self.loftr_fine = LocalFeatureTransformer(config['fine'])
        self.fine_matching = FineMatching()
        self.proj = nn.Sequential(
            nn.Conv2d(256, 128, kernel_size=(1, 1)),
            nn.GroupNorm(32, 128)
        )

        self.matcher_head = nn.ModuleList([
            self.pos_encoding, 
            # self.depth_predictor, 
            self.loftr_coarse, 
            self.coarse_matching, 
            self.fine_preprocess, 
            self.loftr_fine, 
            self.fine_matching, 
            self.proj])
        
    def forward(self, data):
        """
        Update:
            data (dict): {
                'image0': (torch.Tensor): (N, 1, H, W)
                'image1': (torch.Tensor): (N, 1, H, W)
                'mask0'(optional) : (torch.Tensor): (N, H, W) '0' indicates a padded position
                'mask1'(optional) : (torch.Tensor): (N, H, W)
            }
        """
        # 1. Local Feature CNN
        data.update({
            'bs': data['image0'].size(0),  # Batch size
            'hw0_i': data['image0'].shape[2:],  # (H, W)
            'hw1_i': data['image1'].shape[2:]
        })
        
        if data['hw0_i'] == data['hw1_i']: # faster & better BN convergence
            # get coarse and fine level features from backbone
            feats_c, feats_m, feats_f = self.backbone(torch.cat([data['image0'], data['image1']], dim=0))
            # remember to split into two sets
            (feat_c0, feat_c1), (feat_m0, feat_m1), (feat_f0, feat_f1) = feats_c.split(data['bs']), feats_m.split(data['bs']), feats_f.split(data['bs'])
        else: # handle different input shapes
            (feat_c0, feat_m0, feat_f0), (feat_c1, feat_m1, feat_f1) = self.backbone(data['image0']), self.backbone(data['image1'])
        
        # record height and width for both coarse and fine level feature maps
        data.update({
            'hw0_c': feat_c0.shape[2:], 'hw1_c': feat_c1.shape[2:],
            'hw0_f': feat_f0.shape[2:], 'hw1_f': feat_f1.shape[2:]
        })

        # debug: for dimension of model set as 128
        feat_c0 = self.proj(feat_c0)
        feat_c1 = self.proj(feat_c1)

        # 1.5. Depth predictor
        # depth_logits0, _, weighted_depth0 = self.depth_predictor([feat_m0, feat_c0], None)
        # depth_logits1, _, weighted_depth1 = self.depth_predictor([feat_m1, feat_c1], None)
        # For debug
        depth_logits0, weighted_depth0 = None, None
        depth_logits1, weighted_depth1 = None, None

        # 2. Get 3D position encoding
        pos_embed0 = self.pos_encoding(data['hw0_i'], data['hw0_c'], data['K0'], data['T_0to1'])
        pos_embed1 = self.pos_encoding(data['hw1_i'], data['hw1_c'], data['K1'])

        # 3. Coarse-level loftr module
        # flatten features
        feat_c0 = rearrange(feat_c0, 'n c h w -> n (h w) c')
        feat_c1 = rearrange(feat_c1, 'n c h w -> n (h w) c')
        pos_embed0 = rearrange(pos_embed0, 'n c h w -> n (h w) c')
        pos_embed1 = rearrange(pos_embed1, 'n c h w -> n (h w) c')
        
        mask_c0 = mask_c1 = None # useful in training
        if 'mask0' in data:
            mask_c0, mask_c1 = data['mask0'].flatten(-2), data['mask1'].flatten(-2)
        feat_c0, feat_c1 = self.loftr_coarse(feat_c0, feat_c1, pos_embed0, pos_embed1, mask_c0, mask_c1)
        
        # 4. match coarse-level
        self.coarse_matching(feat_c0, feat_c1, data, mask_c0=mask_c0, mask_c1=mask_c1, dmap_c0=weighted_depth0, dmap_c1=weighted_depth1)
        
        # 5. fine-level refinement
        feat_f0_unfold, feat_f1_unfold = self.fine_preprocess(feat_f0, feat_f1, feat_c0, feat_c1, data)
        if feat_f0_unfold.size(0) != 0:
            feat_f0_unfold, feat_f1_unfold = self.loftr_fine(feat_f0_unfold, feat_f1_unfold)
            
        # 6. match fine-level
        self.fine_matching(feat_f0_unfold, feat_f1_unfold, data)
        
        return depth_logits0, depth_logits1, weighted_depth0, weighted_depth1