import torch
import gpytorch


class TanimotoKernel(gpytorch.kernels.Kernel):
    has_lengthscale = True

    def forward(self, x1, x2, **params):
        # Ensure non-negative inputs
        if torch.any(x1 < 0) or torch.any(x2 < 0):
            raise ValueError("Tanimoto kernel requires nonnegative feature vectors.")

        # Dummy lengthscale
        self.register_parameter(
            name="raw_lengthscale",
            parameter=torch.nn.Parameter(torch.zeros(1))
        )

        # Compute dot products
        x1x2 = torch.matmul(x1, x2.T)
        x1_sq = torch.sum(x1**2, dim=-1, keepdim=True)
        x2_sq = torch.sum(x2**2, dim=-1, keepdim=True).T

        # Compute Tanimoto similarity
        denominator = x1_sq + x2_sq - x1x2
        K = x1x2 / denominator.clamp(min=1e-10)  # Avoid division by zero

        return K


if __name__ == "__main__":
    # Example usage:
    x = torch.tensor([[1.0, 2.0], [0.0, 1.0], [2.0, 3.0]])
    kernel = TanimotoKernel()
    K_matrix = kernel(x, x).evaluate()
    print(K_matrix)
