from __future__ import print_function, division, absolute_import
import argparse
import os
import time

import torch
import torch.nn as nn
import torch.optim
import torch.utils.data

import pretrainedmodels
import pretrainedmodels.utils

from utils import SubsetImageNet

model_names = list(
    sorted(name for name in pretrainedmodels.__dict__
        if not name.startswith("__")
        and name.islower()
        and callable(pretrainedmodels.__dict__[name]))
    ) + ['tf_inception_v3', 'tf_inception_v4', 'tf_resnet_v2_152'] \
        + [ 'tf_ens3_adv_inc_v3', 'tf_ens4_adv_inc_v3', 'tf_ens_adv_inc_res_v2' ]

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
parser.add_argument('--input_dir', metavar='DIR', default="path_to_dataset",
                    help='path to dataset')
parser.add_argument('--arch', '-a', metavar='ARCH', default='nasnetamobile',
                    choices=model_names,
                    help='model architecture: ' +
                         ' | '.join(model_names) +
                         ' (default: fbresnet152)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('-b', '--batch-size', default=256, type=int,
                    metavar='N', help='mini-batch size (default: 256)')
parser.add_argument('--print-freq', '-p', default=10, type=int,
                    metavar='N', help='print frequency (default: 10)')
parser.add_argument('--pretrained', default='imagenet', help='use pre-trained model')
parser.add_argument('--do-not-preserve-aspect-ratio',
                    dest='preserve_aspect_ratio',
                    help='do not preserve the aspect ratio when resizing an image',
                    action='store_false')
parser.set_defaults(preserve_aspect_ratio=True)
best_prec1 = 0

from utils import get_eval_model 

def main():
    global args, best_prec1
    args = parser.parse_args()
    print(args)

    # create model
    # print("=> creating model '{}'".format(args.arch))
    model, val_tf = get_eval_model(args.arch) 

    # Data loading code
    print(val_tf)
    valdir = os.path.join(args.input_dir)

    val_set = SubsetImageNet(root=valdir, transform=val_tf)
    val_loader = torch.utils.data.DataLoader(
        val_set,
        batch_size=args.batch_size, shuffle=False,
        num_workers=args.workers, pin_memory=True)

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda()
    model = model.eval().cuda()

    validate(val_loader, model, criterion, args.arch)


from torchmetrics import Accuracy

def validate(val_loader, model, criterion, arch):
    top1 = Accuracy().cuda()
    with torch.no_grad():
        batch_time = AverageMeter()

        # switch to evaluate mode
        model.eval()

        end = time.time()
        for i, raw_data in enumerate(val_loader):
            input = raw_data[0]
            target = raw_data[1]
            target = target.cuda()
            input = input.cuda()

            # compute output
            output = model(input)
            if arch.startswith('tf'):
                output = output[0]
                target += 1

            # measure accuracy and record loss
            top1.update(output, target)

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()


        print('{arch} * Acc@1 {top1:.3f}'
              .format(arch=arch, top1=top1.compute().item() * 100))


class AverageMeter(object):
    """Computes and stores the average and current value"""

    def __init__(self):
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count


def accuracy(output, target, topk=(1,)):
    """Computes the precision@k for the specified values of k"""
    maxk = max(topk)
    batch_size = target.size(0)

    _, pred = output.topk(maxk, 1, True, True)
    pred = pred.t()
    correct = pred.eq(target.view(1, -1).expand_as(pred))

    res = []
    for k in topk:
        correct_k = correct[:k].reshape(-1).float().sum(0)
        res.append(correct_k.mul_(100.0 / batch_size))
    return res


if __name__ == '__main__':
    main()
