import os
import time
import json
import argparse
import subprocess

from multiprocessing.pool import ThreadPool


def print_with_time(content):
    print(time.strftime("%Y-%m-%d %H:%M:%S UTC %z", time.gmtime()), flush=True)
    print(content, '\n', flush=True)


def open_and_wait_timeout(cmd, logfile, timeout=86400):
    with open(logfile, 'w') as f:
        process = subprocess.Popen(cmd, shell=True, stdout=f, stderr=f)
        print_with_time('open cmd: {}'.format(cmd))
        try:
            process.wait(timeout=timeout)
            print_with_time('finished cmd: {}'.format(cmd))
        except subprocess.TimeoutExpired:
            process.terminate()
            print_with_time('timeout cmd: {}'.format(cmd))


parser = argparse.ArgumentParser(description='testing pcstable for datasets')
parser.add_argument('--datasets', type=str, default=None, help='datasets info, in .json file')
parser.add_argument('--parallel', type=int, default=1, help='eval number of dataset in the same time')
parser.add_argument('--output', type=str, default=None, help='output dir path')
args = parser.parse_args()

with open(args.datasets) as f:
    datasets = json.load(f)

if not os.path.exists(os.path.join(args.output, 'cache')):
    os.makedirs(os.path.join(args.output, 'cache'))

tp = ThreadPool(args.parallel)
for dataset in datasets:
    dataset = datasets[dataset]
    if os.path.exists(os.path.join(args.output, 'cache', '{}_metrics.json'.format(dataset['name']))): continue

    cmd = 'python -m algorithms.pcstable_default_runner' + \
        ' --data {}'.format(os.path.abspath(dataset['data'])) + \
        ' --truth {}'.format(os.path.abspath(dataset['graph'])) + \
        ' --est {}'.format(os.path.abspath(os.path.join(args.output, 'cache', '{}_est.json'.format(dataset['name'])))) + \
        ' --metrics {}'.format(os.path.abspath(os.path.join(args.output, 'cache', '{}_metrics.json'.format(dataset['name']))))
    tp.apply_async(open_and_wait_timeout, 
                   (cmd, os.path.join(args.output, 'cache', '{}_logger.log'.format(dataset['name'])), dataset['timeout']))
tp.close()
tp.join()

result = {}
for dataset in datasets:
    dataset = datasets[dataset]

    if not os.path.exists(os.path.join(args.output, 'cache', '{}_metrics.json'.format(dataset['name']))):
        print_with_time('file not exists - {}'.format(os.path.join(args.output, 'cache', '{}_metrics.json'.format(dataset['name']))))
        dataset['success'] = False
    else:
        dataset['success'] = True

        with open(os.path.join(args.output, 'cache', '{}_metrics.json'.format(dataset['name']))) as f:
            sub_metrics = json.load(f)
        
        dataset = {**dataset, **sub_metrics}
    
    result[dataset['name']] = dataset
with open(os.path.join(args.output, 'result.json'), 'w') as f:
    json.dump(result, f, indent='\t')

    


