from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(on b1 b2)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class BlocksworldHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Blocksworld domain.

    # Summary
    This heuristic estimates the number of actions required to reach the goal state by counting the number of blocks that are not in their correct position or not correctly stacked. It also considers the number of blocks that need to be moved to achieve the correct stacking order.

    # Assumptions:
    - The goal state specifies the desired stacking order of blocks.
    - Blocks can only be moved one at a time.
    - The heuristic does not need to be admissible, so it can overestimate the number of actions.

    # Heuristic Initialization
    - Extract the goal conditions to determine the desired stacking order.
    - Identify the initial state of the blocks to compare against the goal state.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current stacking order of blocks in the state.
    2. Compare the current stacking order with the goal stacking order.
    3. Count the number of blocks that are not in their correct position or not correctly stacked.
    4. For each block that is not in its correct position, estimate the number of actions required to move it to the correct position.
    5. Sum the estimated actions to get the heuristic value.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and initial state."""
        self.goals = task.goals
        self.static = task.static

        # Extract the goal stacking order
        self.goal_stack = {}
        for goal in self.goals:
            if match(goal, "on", "*", "*"):
                _, block, under_block = get_parts(goal)
                self.goal_stack[block] = under_block
            elif match(goal, "on-table", "*"):
                _, block = get_parts(goal)
                self.goal_stack[block] = None

    def __call__(self, node):
        """Estimate the number of actions required to reach the goal state."""
        state = node.state

        # Initialize the heuristic value
        heuristic_value = 0

        # Track the current stacking order
        current_stack = {}
        for fact in state:
            if match(fact, "on", "*", "*"):
                _, block, under_block = get_parts(fact)
                current_stack[block] = under_block
            elif match(fact, "on-table", "*"):
                _, block = get_parts(fact)
                current_stack[block] = None

        # Compare current stacking with goal stacking
        for block, goal_under_block in self.goal_stack.items():
            if block not in current_stack or current_stack[block] != goal_under_block:
                heuristic_value += 1

        return heuristic_value
