from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(on b1 b2)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class BlocksworldHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Blocksworld domain.

    # Summary
    This heuristic estimates the number of actions required to reach the goal state by counting the number of blocks that are not in their correct position or not correctly stacked. It also considers the number of blocks that need to be moved to achieve the correct stacking.

    # Assumptions:
    - The goal state specifies the desired stacking of blocks.
    - Blocks can only be moved one at a time.
    - The heuristic does not need to be admissible, so it can overestimate the number of actions.

    # Heuristic Initialization
    - Extract the goal conditions for each block, specifically the desired `on` and `on-table` relationships.
    - Identify the static facts (if any) that are always true in the domain.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each block, check if it is in its correct position as specified by the goal.
    2. If a block is not in its correct position, increment the heuristic value by 1.
    3. If a block is on top of another block that is not its correct base, increment the heuristic value by 1.
    4. If a block is on the table but should be on another block, increment the heuristic value by 1.
    5. If a block is on another block but should be on the table, increment the heuristic value by 1.
    6. The total heuristic value is the sum of all these increments.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts (not used in this heuristic).

        # Extract the goal conditions for each block.
        self.goal_on = {}
        self.goal_on_table = set()
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "on":
                block, base = args
                self.goal_on[block] = base
            elif predicate == "on-table":
                block = args[0]
                self.goal_on_table.add(block)

    def __call__(self, node):
        """Estimate the number of actions required to reach the goal state."""
        state = node.state  # Current world state.

        # Track the current positions of blocks.
        current_on = {}
        current_on_table = set()
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "on":
                block, base = args
                current_on[block] = base
            elif predicate == "on-table":
                block = args[0]
                current_on_table.add(block)

        total_cost = 0  # Initialize the heuristic cost.

        # Check each block's position against the goal.
        for block in self.goal_on_table:
            if block not in current_on_table:
                total_cost += 1  # Block should be on the table but isn't.

        for block, base in self.goal_on.items():
            if block in current_on:
                if current_on[block] != base:
                    total_cost += 1  # Block is on the wrong base.
            else:
                total_cost += 1  # Block is not on any base (should be on a specific one).

        return total_cost
