from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at tray1 kitchen)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class ChildsnackHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Childsnacks domain.

    # Summary
    This heuristic estimates the number of actions required to serve all children with sandwiches, considering the constraints of gluten allergies and the availability of ingredients.

    # Assumptions
    - Each child must be served exactly one sandwich.
    - Children with gluten allergies must be served gluten-free sandwiches.
    - Sandwiches must be made from available bread and content portions in the kitchen.
    - Sandwiches must be placed on trays and moved to the children's locations.

    # Heuristic Initialization
    - Extract the goal conditions (all children must be served).
    - Extract static facts (e.g., which children are allergic to gluten, which bread and content portions are gluten-free).
    - Identify the initial state (e.g., which ingredients are available, which trays are in the kitchen).

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the number of children who are not yet served.
    2. For each unserved child:
       - If the child is allergic to gluten, ensure that a gluten-free sandwich is available or can be made.
       - If the child is not allergic to gluten, any sandwich can be used.
    3. Count the number of sandwiches that need to be made:
       - For gluten-free sandwiches, ensure that both gluten-free bread and content portions are available.
       - For regular sandwiches, ensure that any bread and content portions are available.
    4. Count the number of sandwiches that need to be placed on trays:
       - Each sandwich must be placed on a tray before it can be served.
    5. Count the number of trays that need to be moved to the children's locations:
       - Each tray must be moved to the location where the child is waiting.
    6. Sum the total number of actions required:
       - Making sandwiches, placing them on trays, and moving trays to serve the children.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts.

        # Extract information about children and their allergies.
        self.allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "allergic_gluten", "*")
        }
        self.not_allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "not_allergic_gluten", "*")
        }

        # Extract information about gluten-free ingredients.
        self.gluten_free_bread = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_bread", "*")
        }
        self.gluten_free_content = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_content", "*")
        }

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Count the number of children who are not yet served.
        unserved_children = {
            get_parts(fact)[1] for fact in self.goals if match(fact, "served", "*")
        } - {
            get_parts(fact)[1] for fact in state if match(fact, "served", "*")
        }

        # Initialize the heuristic cost.
        total_cost = 0

        # For each unserved child, estimate the actions required to serve them.
        for child in unserved_children:
            if child in self.allergic_children:
                # Gluten-free sandwich required.
                # Check if a gluten-free sandwich is already on a tray.
                gluten_free_sandwich_on_tray = any(
                    match(fact, "ontray", "*", "*") and match(fact, "*", "*", "*", "no_gluten_sandwich", "*")
                    for fact in state
                )
                if not gluten_free_sandwich_on_tray:
                    # Need to make a gluten-free sandwich.
                    total_cost += 1  # Make sandwich.
                    total_cost += 1  # Put on tray.
            else:
                # Any sandwich can be used.
                # Check if any sandwich is already on a tray.
                sandwich_on_tray = any(match(fact, "ontray", "*", "*") for fact in state)
                if not sandwich_on_tray:
                    # Need to make a sandwich.
                    total_cost += 1  # Make sandwich.
                    total_cost += 1  # Put on tray.

            # Move the tray to the child's location.
            total_cost += 1  # Move tray.

            # Serve the sandwich.
            total_cost += 1  # Serve sandwich.

        return total_cost
