from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at tray1 kitchen)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class ChildsnackHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Childsnacks domain.

    # Summary
    This heuristic estimates the number of actions required to serve all children with sandwiches, considering the constraints of gluten allergies and the availability of ingredients.

    # Assumptions:
    - Each child must be served exactly one sandwich.
    - Children with gluten allergies must be served gluten-free sandwiches.
    - Sandwiches must be made from available bread and content portions in the kitchen.
    - Sandwiches must be placed on trays and moved to the appropriate locations before serving.

    # Heuristic Initialization
    - Extract the goal conditions (served children) and static facts (allergic children, waiting locations, etc.).
    - Identify the number of children, trays, and available ingredients.

    # Step-By-Step Thinking for Computing Heuristic
    1. Count the number of children who are not yet served.
    2. For each unserved child:
       - If the child is allergic to gluten, ensure a gluten-free sandwich is available.
       - If the child is not allergic, any sandwich can be used.
    3. Count the number of sandwiches that need to be made:
       - For each required sandwich, check if the necessary ingredients are available in the kitchen.
    4. Count the number of trays needed to transport the sandwiches:
       - Each tray can carry multiple sandwiches, but each sandwich must be placed on a tray.
    5. Estimate the number of actions required:
       - Making sandwiches: 1 action per sandwich.
       - Placing sandwiches on trays: 1 action per sandwich.
       - Moving trays to the correct locations: 1 action per tray.
       - Serving sandwiches: 1 action per child.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts.

        # Extract information about allergic children and waiting locations.
        self.allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "allergic_gluten", "*")
        }
        self.waiting_locations = {
            get_parts(fact)[1]: get_parts(fact)[2] for fact in self.static if match(fact, "waiting", "*", "*")
        }

    def __call__(self, node):
        """Estimate the number of actions required to serve all children."""
        state = node.state

        # Count the number of children who are not yet served.
        unserved_children = {
            child for child in self.waiting_locations if f"(served {child})" not in state
        }

        # Count the number of sandwiches that need to be made.
        sandwiches_needed = 0
        for child in unserved_children:
            if child in self.allergic_children:
                # Need a gluten-free sandwich.
                sandwiches_needed += 1
            else:
                # Any sandwich will do.
                sandwiches_needed += 1

        # Count the number of sandwiches already made and on trays.
        sandwiches_on_trays = sum(1 for fact in state if match(fact, "ontray", "*", "*"))

        # Count the number of trays available.
        trays_available = sum(1 for fact in state if match(fact, "at", "tray*", "*"))

        # Estimate the number of actions required.
        total_cost = 0

        # Actions to make sandwiches.
        total_cost += max(0, sandwiches_needed - sandwiches_on_trays)

        # Actions to place sandwiches on trays.
        total_cost += sandwiches_needed

        # Actions to move trays to the correct locations.
        total_cost += trays_available

        # Actions to serve sandwiches.
        total_cost += len(unserved_children)

        return total_cost
