from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at_kitchen_bread bread1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class ChildsnackHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Childsnacks domain.

    # Summary
    This heuristic estimates the number of actions required to serve all children with sandwiches, considering the constraints of gluten allergies and the availability of ingredients.

    # Assumptions
    - Each child must be served exactly one sandwich.
    - Children with gluten allergies must be served gluten-free sandwiches.
    - Sandwiches must be made from available bread and content portions in the kitchen.
    - Sandwiches must be placed on trays and moved to the correct location before serving.

    # Heuristic Initialization
    - Extract the goal conditions (served children) and static facts (allergic children, gluten-free ingredients, etc.).
    - Identify the number of children, sandwiches, trays, and ingredients available.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the number of children who still need to be served.
    2. Determine the number of sandwiches that need to be made, considering gluten-free requirements.
    3. Check if the required sandwiches are already made and placed on trays.
    4. Estimate the number of actions required to:
       - Make sandwiches (if not already made).
       - Place sandwiches on trays (if not already on trays).
       - Move trays to the correct locations (if not already there).
       - Serve sandwiches to children.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts.

        # Extract information about allergic and non-allergic children.
        self.allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "allergic_gluten", "*")
        }
        self.non_allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "not_allergic_gluten", "*")
        }

        # Extract information about gluten-free ingredients.
        self.gluten_free_bread = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_bread", "*")
        }
        self.gluten_free_content = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_content", "*")
        }

    def __call__(self, node):
        """Estimate the number of actions required to serve all children."""
        state = node.state

        # Count the number of children who still need to be served.
        unserved_children = {
            get_parts(fact)[1] for fact in self.goals if match(fact, "served", "*")
        } - {
            get_parts(fact)[1] for fact in state if match(fact, "served", "*")
        }

        # Initialize the heuristic cost.
        total_cost = 0

        # For each unserved child, estimate the actions required to serve them.
        for child in unserved_children:
            if child in self.allergic_children:
                # Gluten-free sandwich required.
                # Check if a gluten-free sandwich is already on a tray.
                gluten_free_sandwich_on_tray = any(
                    match(fact, "ontray", "*", "*") and match(fact, "no_gluten_sandwich", "*")
                    for fact in state
                )
                if not gluten_free_sandwich_on_tray:
                    # Need to make a gluten-free sandwich.
                    total_cost += 1  # make_sandwich_no_gluten action.
                    total_cost += 1  # put_on_tray action.
            else:
                # Regular sandwich is acceptable.
                # Check if any sandwich is already on a tray.
                sandwich_on_tray = any(match(fact, "ontray", "*", "*") for fact in state)
                if not sandwich_on_tray:
                    # Need to make a sandwich.
                    total_cost += 1  # make_sandwich action.
                    total_cost += 1  # put_on_tray action.

            # Move the tray to the child's location (if not already there).
            tray_at_location = any(
                match(fact, "at", "*", get_parts(fact)[2]) and match(fact, "waiting", child, "*")
                for fact in state
            )
            if not tray_at_location:
                total_cost += 1  # move_tray action.

            # Serve the sandwich.
            total_cost += 1  # serve_sandwich or serve_sandwich_no_gluten action.

        return total_cost
