from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(robot-at robot1 tile_0_1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class FloortileHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Floortile domain.

    # Summary
    This heuristic estimates the number of actions required to paint all tiles according to the goal conditions.
    It considers the following:
    - The number of tiles that still need to be painted.
    - The distance between the robot's current position and the tiles that need to be painted.
    - The number of color changes required for the robot to paint the tiles with the correct colors.

    # Assumptions
    - The robot can only paint tiles adjacent to its current position (up, down, left, right).
    - The robot must change its color if it needs to paint a tile with a different color than the one it currently holds.
    - The robot can move freely between tiles as long as they are clear.

    # Heuristic Initialization
    - Extract the goal conditions for each tile (i.e., the required color for each tile).
    - Extract the static facts (e.g., adjacency relationships between tiles) to determine movement costs.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the tiles that still need to be painted (i.e., tiles that are either clear or painted with the wrong color).
    2. For each such tile, calculate the Manhattan distance from the robot's current position to the tile.
    3. If the robot needs to change its color to paint the tile, add an additional cost for the color change.
    4. Sum the costs for all tiles to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal conditions for each tile.
        - Static facts (adjacency relationships between tiles).
        """
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts.

        # Extract goal conditions for each tile.
        self.goal_paintings = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "painted":
                tile, color = args
                self.goal_paintings[tile] = color

        # Extract adjacency relationships between tiles.
        self.adjacent = {}
        for fact in self.static:
            predicate, *args = get_parts(fact)
            if predicate in ["up", "down", "left", "right"]:
                tile1, tile2 = args
                if tile1 not in self.adjacent:
                    self.adjacent[tile1] = []
                self.adjacent[tile1].append(tile2)

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Get the robot's current position and color.
        robot_position = None
        robot_color = None
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "robot-at":
                robot, position = args
                robot_position = position
            elif predicate == "robot-has":
                robot, color = args
                robot_color = color

        total_cost = 0  # Initialize action cost counter.

        for tile, goal_color in self.goal_paintings.items():
            # Check if the tile is already painted with the correct color.
            is_painted_correctly = f"(painted {tile} {goal_color})" in state
            if is_painted_correctly:
                continue

            # Calculate the Manhattan distance from the robot's current position to the tile.
            if robot_position:
                distance = self.manhattan_distance(robot_position, tile)
            else:
                distance = 0

            # Add the movement cost.
            total_cost += distance

            # Check if the robot needs to change its color to paint the tile.
            if robot_color != goal_color:
                total_cost += 1  # Cost for changing color.

        return total_cost

    def manhattan_distance(self, tile1, tile2):
        """
        Calculate the Manhattan distance between two tiles based on their coordinates.
        Assumes tile names are in the format "tile_X_Y".
        """
        x1, y1 = map(int, tile1.split("_")[1:])
        x2, y2 = map(int, tile2.split("_")[1:])
        return abs(x1 - x2) + abs(y1 - y2)
