from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(robot-at robot1 tile_0_1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class FloortileHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Floortile domain.

    # Summary
    This heuristic estimates the number of actions required to paint all tiles according to the goal conditions.
    It considers the number of tiles that still need to be painted, the current color of the robot, and the distance
    between the robot and the tiles that need to be painted.

    # Assumptions
    - The robot can only paint adjacent tiles (up, down, left, right).
    - The robot must change color if it needs to paint a tile with a different color than the one it currently holds.
    - The robot can move freely between tiles as long as they are clear.

    # Heuristic Initialization
    - Extract the goal conditions for each tile (i.e., which color each tile should be painted).
    - Extract the static facts (e.g., adjacency relationships between tiles) to compute distances.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the tiles that still need to be painted according to the goal conditions.
    2. For each tile that needs to be painted:
        - If the robot is not holding the correct color, add a cost for changing the color.
        - Compute the Manhattan distance between the robot's current position and the tile.
        - Add the distance as the cost for moving the robot to the tile.
        - Add a cost for painting the tile.
    3. Sum the costs for all tiles to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal conditions for each tile.
        - Static facts (adjacency relationships between tiles).
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract goal conditions for each tile.
        self.goal_paintings = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "painted":
                tile, color = args
                self.goal_paintings[tile] = color

        # Extract adjacency relationships between tiles.
        self.adjacent = {}
        for fact in static_facts:
            predicate, *args = get_parts(fact)
            if predicate in ["up", "down", "left", "right"]:
                tile1, tile2 = args
                if tile1 not in self.adjacent:
                    self.adjacent[tile1] = []
                self.adjacent[tile1].append((predicate, tile2))

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Extract the robot's current position and color.
        robot_position = None
        robot_color = None
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "robot-at":
                robot, position = args
                robot_position = position
            elif predicate == "robot-has":
                robot, color = args
                robot_color = color

        total_cost = 0  # Initialize action cost counter.

        for tile, goal_color in self.goal_paintings.items():
            # Check if the tile is already painted correctly.
            if f"(painted {tile} {goal_color})" in state:
                continue

            # If the robot is not holding the correct color, add a cost for changing color.
            if robot_color != goal_color:
                total_cost += 1  # Cost for changing color.

            # Compute the Manhattan distance between the robot and the tile.
            distance = self.compute_manhattan_distance(robot_position, tile)
            total_cost += distance  # Cost for moving to the tile.

            # Add a cost for painting the tile.
            total_cost += 1

        return total_cost

    def compute_manhattan_distance(self, start_tile, goal_tile):
        """
        Compute the Manhattan distance between two tiles based on their coordinates.

        - `start_tile`: The starting tile (e.g., "tile_0_1").
        - `goal_tile`: The goal tile (e.g., "tile_3_4").
        - Returns the Manhattan distance as an integer.
        """
        # Extract coordinates from tile names.
        start_x, start_y = map(int, start_tile.split("_")[1:])
        goal_x, goal_y = map(int, goal_tile.split("_")[1:])

        return abs(start_x - goal_x) + abs(start_y - goal_y)
