from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic


def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()


def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(origin p1 f1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))


class MiconicHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Miconic domain.

    # Summary
    This heuristic estimates the number of actions required to serve all passengers by considering:
    - The current position of the lift.
    - The origin and destination floors of each passenger.
    - Whether passengers are already boarded or served.

    # Assumptions:
    - The lift can move between floors using the `up` and `down` actions.
    - Passengers must be boarded before they can be served.
    - The heuristic does not need to be admissible, so it can overestimate the number of actions.

    # Heuristic Initialization
    - Extract the goal conditions (all passengers must be served).
    - Extract static facts, such as the `above` relationships between floors and the `destin` of each passenger.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current state of the lift and the passengers:
       - Check which passengers are already served (no further actions needed).
       - Check which passengers are boarded (they need to be served at their destination).
       - Check which passengers are still at their origin (they need to be boarded and served).
    2. For each passenger not yet served:
       - Calculate the number of floors the lift must travel to reach their origin floor.
       - Calculate the number of floors the lift must travel to reach their destination floor.
       - Add the cost of boarding and serving the passenger.
    3. Sum the total number of actions required to serve all passengers.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract the destination floors for each passenger.
        self.destin_floors = {
            get_parts(fact)[1]: get_parts(fact)[2]
            for fact in static_facts
            if match(fact, "destin", "*", "*")
        }

        # Extract the `above` relationships between floors.
        self.above_relationships = {
            (get_parts(fact)[1], get_parts(fact)[2])
            for fact in static_facts
            if match(fact, "above", "*", "*")
        }

    def __call__(self, node):
        """Estimate the number of actions required to serve all passengers."""
        state = node.state  # Current world state.

        # Identify the current position of the lift.
        lift_at = None
        for fact in state:
            if match(fact, "lift-at", "*"):
                lift_at = get_parts(fact)[1]
                break

        # Initialize the total cost.
        total_cost = 0

        # Track which passengers are already served.
        served_passengers = {
            get_parts(fact)[1] for fact in state if match(fact, "served", "*")
        }

        # Track which passengers are boarded.
        boarded_passengers = {
            get_parts(fact)[1] for fact in state if match(fact, "boarded", "*")
        }

        # Track the origin floors of passengers not yet boarded.
        origin_floors = {
            get_parts(fact)[1]: get_parts(fact)[2]
            for fact in state
            if match(fact, "origin", "*", "*")
        }

        # For each passenger not yet served, calculate the required actions.
        for passenger, destin_floor in self.destin_floors.items():
            if passenger in served_passengers:
                continue  # Passenger is already served.

            # Calculate the cost to board the passenger.
            if passenger not in boarded_passengers:
                origin_floor = origin_floors[passenger]
                # Calculate the number of floors to travel to the origin floor.
                total_cost += self._calculate_travel_cost(lift_at, origin_floor)
                # Add the cost of boarding the passenger.
                total_cost += 1  # Board action.
                lift_at = origin_floor  # Update lift position.

            # Calculate the cost to serve the passenger.
            total_cost += self._calculate_travel_cost(lift_at, destin_floor)
            total_cost += 1  # Depart action.
            lift_at = destin_floor  # Update lift position.

        return total_cost

    def _calculate_travel_cost(self, current_floor, target_floor):
        """
        Calculate the number of actions required to move the lift from `current_floor` to `target_floor`.

        - `current_floor`: The current floor of the lift.
        - `target_floor`: The target floor.
        - Returns the number of `up` or `down` actions required.
        """
        if current_floor == target_floor:
            return 0

        # Determine the direction of travel.
        if (current_floor, target_floor) in self.above_relationships:
            return 1  # One `up` action.
        else:
            return 1  # One `down` action.
