from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(origin p1 f1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class MiconicHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Miconic domain.

    # Summary
    This heuristic estimates the number of actions required to serve all passengers by considering:
    - The current position of the lift.
    - The origin and destination floors of each passenger.
    - Whether passengers are already boarded or served.

    # Assumptions:
    - The lift can move between floors in a single action.
    - Passengers must be boarded before they can be served.
    - The heuristic does not need to be admissible, so it can overestimate the number of actions.

    # Heuristic Initialization
    - Extract the goal conditions (all passengers must be served).
    - Extract static facts, such as the `above` relationships between floors and the `destin` of each passenger.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current state of the lift and the passengers:
       - Check if passengers are already served or boarded.
       - Determine the current floor of the lift.
    2. For each passenger not yet served:
       - If the passenger is not boarded, estimate the cost to board them:
         - Move the lift to their origin floor.
         - Board the passenger.
       - If the passenger is boarded, estimate the cost to serve them:
         - Move the lift to their destination floor.
         - Depart the passenger.
    3. Sum the estimated actions for all passengers to get the total heuristic value.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract the `above` relationships between floors.
        self.above = {
            (get_parts(fact)[1], get_parts(fact)[2])
            for fact in static_facts
            if match(fact, "above", "*", "*")
        }

        # Extract the destination floors for each passenger.
        self.destin = {
            get_parts(fact)[1]: get_parts(fact)[2]
            for fact in static_facts
            if match(fact, "destin", "*", "*")
        }

    def __call__(self, node):
        """Estimate the number of actions required to serve all passengers."""
        state = node.state  # Current world state.

        # Check if the goal is already reached.
        if self.goals <= state:
            return 0

        # Extract the current floor of the lift.
        lift_at = None
        for fact in state:
            if match(fact, "lift-at", "*"):
                lift_at = get_parts(fact)[1]
                break

        total_cost = 0  # Initialize the heuristic cost.

        # Iterate over all passengers to estimate the cost.
        for fact in state:
            if match(fact, "origin", "*", "*"):
                passenger, origin_floor = get_parts(fact)[1], get_parts(fact)[2]

                # Skip if the passenger is already served.
                if f"(served {passenger})" in state:
                    continue

                # If the passenger is not boarded, estimate the cost to board them.
                if f"(boarded {passenger})" not in state:
                    # Move the lift to the origin floor.
                    if lift_at != origin_floor:
                        total_cost += 1
                    # Board the passenger.
                    total_cost += 1

                # Estimate the cost to serve the passenger.
                destin_floor = self.destin[passenger]
                if lift_at != destin_floor:
                    total_cost += 1
                # Depart the passenger.
                total_cost += 1

        return total_cost
